/*
   Niteto ChipChecker - Tests old DRAM 64kx4 Chips
   Copyright (C) 2017 Dominik Tonn (nick@niteto.de)
   This is for Teensy 3.2

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __MK20DX256__
#error Not Teensy 3.2 MCU, make sure you get a version 3.2, because the 3.1 ist NOT 5V tolerant
#endif

#include <Streaming.h>

#define Version "v0.100"

// These are the physikal pin numbers of the chip
// they will be translated later to the real gpio
// pins they are attached to.
// defines for TMS 4464 and compatible 64kx4 chips
#define RA0 14
#define RA1 13
#define RA2 12
#define RA3 11
#define RA4 8
#define RA5 7
#define RA6 6
#define RA7 10
#define RAS 5
#define CAS 16
#define IO1 2
#define IO2 3
#define IO3 15
#define IO4 17
#define _OE  1
#define _WE  4
#define DIL18 18

String inputString = "";
String cmd, param;
uint32_t zeit;
uint32_t tRefresh;

uint8_t adr[8] {RA0, RA1, RA2, RA3, RA4, RA5, RA6, RA7};
uint8_t chip[24] {11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0, 23, 22, 21, 20, 19, 18, 17, 16, 15, 14, 13, 12};

// translates a chip pin to the gpio pin
// dil - chip package size
// pin - chip pin number, e.g. 1-18 for DIL18
// returns the gpio pin of the teensy
uint8_t translatePin(uint8_t dil, uint8_t pin)
{
  pin--;
  if (pin < dil / 2)
    return chip[pin];
  else
    return chip[(24 - dil + pin)];
}

// sets up the pinmodes for the 4464 dram chip
void setupDram()
{
  pinMode(translatePin(DIL18, CAS), OUTPUT);
  digitalWrite(translatePin(DIL18, CAS), HIGH);
  pinMode(translatePin(DIL18, RAS), OUTPUT);
  digitalWrite(translatePin(DIL18, RAS), HIGH);
  pinMode(translatePin(DIL18, _WE), OUTPUT);
  digitalWrite(translatePin(DIL18, _WE), HIGH);
  pinMode(translatePin(DIL18, _OE), OUTPUT);
  digitalWrite(translatePin(DIL18, _OE), HIGH);
  for (int i = 0; i < 8; i++)
    pinMode(translatePin(DIL18, adr[i]), OUTPUT);
}

// puts <adress> on the pins RA0-RA7
void setAdr(uint8_t adress)
{
  for (int i = 0; i < 8; i++)
  {
    digitalWriteFast(translatePin(DIL18, adr[i]), adress & 0x01);
    adress = adress >> 1;
  }
}

// reads from <row>, <col>
// returns 4bits
uint8_t readAdr(uint8_t row, uint8_t col)
{
  pinMode(translatePin(DIL18, IO1), INPUT_PULLUP);
  pinMode(translatePin(DIL18, IO2), INPUT_PULLUP);
  pinMode(translatePin(DIL18, IO3), INPUT_PULLUP);
  pinMode(translatePin(DIL18, IO4), INPUT_PULLUP);
  setAdr(row);
  digitalWriteFast(translatePin(DIL18, RAS), LOW);
  digitalWriteFast(translatePin(DIL18, _WE), HIGH);
  setAdr(col);
  digitalWriteFast(translatePin(DIL18, CAS), LOW);
  uint8_t data = 0;
  digitalWrite(translatePin(DIL18, _OE), LOW);
  data = digitalReadFast(translatePin(DIL18, IO4)) * 8 +
         digitalReadFast(translatePin(DIL18, IO3)) * 4 +
         digitalReadFast(translatePin(DIL18, IO2)) * 2 +
         digitalReadFast(translatePin(DIL18, IO1)) * 1;
  digitalWriteFast(translatePin(DIL18, CAS), HIGH);
  digitalWriteFast(translatePin(DIL18, RAS), HIGH);
  digitalWriteFast(translatePin(DIL18, _OE), HIGH);
  return data;
}

// write lower nibble of <data> to <row>, <col>
void writeAdr(uint8_t row, uint8_t col, uint8_t data)
{
  pinMode(translatePin(DIL18, IO1), OUTPUT);
  pinMode(translatePin(DIL18, IO2), OUTPUT);
  pinMode(translatePin(DIL18, IO3), OUTPUT);
  pinMode(translatePin(DIL18, IO4), OUTPUT);

  setAdr(row);

  digitalWriteFast(translatePin(DIL18, RAS), LOW);
  digitalWriteFast(translatePin(DIL18, _WE), LOW);

  digitalWriteFast(translatePin(DIL18, IO4), (data >> 3) & 0x01);
  digitalWriteFast(translatePin(DIL18, IO3), (data >> 2) & 0x01);
  digitalWriteFast(translatePin(DIL18, IO2), (data >> 1) & 0x01);
  digitalWriteFast(translatePin(DIL18, IO1), (data) & 0x01);

  setAdr(col);

  digitalWriteFast(translatePin(DIL18, CAS), LOW);

  digitalWriteFast(translatePin(DIL18, _WE), HIGH);
  digitalWriteFast(translatePin(DIL18, CAS), HIGH);
  digitalWriteFast(translatePin(DIL18, RAS), HIGH);
  /*
    pinMode(translatePin(DIL18, IO1), INPUT_PULLUP);
    pinMode(translatePin(DIL18, IO2), INPUT_PULLUP);
    pinMode(translatePin(DIL18, IO3), INPUT_PULLUP);
    pinMode(translatePin(DIL18, IO4), INPUT_PULLUP);
  */
}

// tests for stuck databus IO lines
boolean testDatabus()
{
  boolean fail = false;
  uint8_t comp;
  for (int i = 0; i < 16; i++)
  {
    writeAdr(0, 0, i);
    comp = readAdr(0, 0);
    if (comp != i) fail = true;
  }
  if (fail) {
    writeAdr(0, 0, 0x0f);
    uint8_t stuckLow = (~readAdr(0, 0)) & 0x0f;
    writeAdr(0, 0, 0x00);
    uint8_t stuckHigh = readAdr(0, 0) & 0x0f;
    if (stuckLow != 0) {
      Serial.print("IO Lines[1..4] stuck Low: ");
      for (int i = 0; i < 4; i++) {
        if ((stuckLow & 0x01) == 1)
          Serial.printf("IO[%d] ", i + 1);
        stuckLow = stuckLow >> 1;
      }
      Serial.println();
    }
    if (stuckHigh != 0) {
      Serial.print("IO Lines[1..4] stuck High: ");
      for (int i = 0; i < 4; i++) {
        if ((stuckHigh & 0x01) == 1)
          Serial.printf("IO[%d] ", i + 1);
        stuckHigh = stuckHigh >> 1;
      }
      Serial.println();
    }
  }
  return !fail;
}

//tests for stuck adressbus lines
int16_t testAdressbus()
{
  for (uint16_t i = 1; i < 256; i <<= 1)
    writeAdr(i, 0, 0x0a);

  writeAdr(0, 0, 0x05);
  for (uint16_t i = 1; i < 256; i <<= 1)
    if (readAdr(i, 0) != 0x0a) return i;
  writeAdr(0, 0, 0x0a);

  for (uint16_t i = 1; i < 256; i <<= 1)
  {
    writeAdr(i, 0, 0x05);
    if (readAdr(0, 0) != 0x0a) return i;
    for (uint16_t j = 1; j < 256; j <<= 1)
      if ((readAdr(j, 0) != 0x0a) && j != i) return i;
    writeAdr(i, 0, 0x0a);
  }
  return -1;
}

//tests complete memory
int16_t testMem()
{
  int16_t result = -1;
  for (uint16_t row = 0; row < 256; row++)
  {
    if (row % 32 == 0) {
      if (row != 0) Serial.println();
      Serial.printf("Testing: 0x%.4x-0x%.4x ", row * 256, row * 256 + 512 * 16 - 1);
    }
    else
      Serial.print(".");
    for (uint8_t pattern = 0; pattern < 16; pattern++)
    {
      for (uint16_t col = 0; col < 256; col++)
        writeAdr(row, col, pattern);
      for (uint16_t col = 0; col < 256; col++)
        if (readAdr(row, col) != pattern) {
          Serial.printf("\nFail at: 0x%.4x\n", row * 256 + col);
          result = row * 256 + col;
        }
    }
  }
  Serial.println();
  return result;
}

// refresh <row>
void refresh(uint8_t row)
{
  setAdr(row);
  digitalWriteFast(translatePin(DIL18, RAS), LOW);
  digitalWriteFast(translatePin(DIL18, RAS), HIGH);
}

/*
   reads command from the serial port
*/
void getCommand()
{
  inputString = "";
  while (Serial.available())
  {
    char inChar = (char)Serial.read();
    inputString += inChar;
    if (inChar == '\n') {
      break;
    }
  }
}

/*
   execute command
*/
void doCommand()
{
  inputString.replace((char)10, (char)0);
  inputString.replace((char)13, (char)0);
  cmd = inputString;
  param = inputString;
  if (inputString.indexOf(" ") > 0) {
    cmd.remove(inputString.indexOf(" "), inputString.length());
    param.remove(0, inputString.indexOf(" "));
  } else {
    cmd = inputString;
    param = "";
  }
  cmd.trim();
  param.trim();
  //    Serial.printf("cmd: '%s' param: '%s'\n",cmd.c_str(),param.c_str());
  //    Serial.println(cmd);
  //    Serial.println(param);
  if (cmd == "help") {
    Serial << "-----------------------------------------------------------------------------\n";
    Serial << "ChipChecker " << Version << " - Copyright (C) 2017 Dominik Tonn (nick@niteto.de)\n";
    Serial << "-----------------------------------------------------------------------------\n";
    Serial << "data - tests databus for stuck lines\n";
    Serial << "adr  - tests adressbusfor stuck lines\n";
    Serial << "mem  - tests full memory range\n";
    Serial << "full - databus, adressbus and full memory test\n";
    Serial << "time - time the teensy took for a full refresh cycle\n";
  }
  if (cmd == "time") {
    Serial << "Refresh Time: " << tRefresh << "\n";
  }
  if (cmd == "full") {
    Serial.print("Testing DataBus: ");
    if (testDatabus()) Serial.println("OK");
    else Serial.println("FAIL");

    Serial.print("Testing AdressBus: ");
    int16_t result = testAdressbus();
    if (result == -1) Serial.println("OK");
    else Serial.printf("FAIL at: %d\n", result);

    Serial.println("Testing Memory:");
    result = testMem();
    if (result == -1) Serial.println("Memory OK");
    else Serial.println("Memory FAIL");
  }
  if (cmd == "data") {
    Serial.print("Testing DataBus: ");
    if (testDatabus()) Serial.println("OK");
    else Serial.println("FAIL");
  }
  if (cmd == "adr") {
    Serial.print("Testing AdressBus: ");
    int16_t result = testAdressbus();
    if (result == -1) Serial.println("OK");
    else Serial.printf("FAIL at: %d\n", result);
  }
  if (cmd == "mem") {
    Serial.println("Testing Memory:");
    int16_t result = testMem();
    if (result == -1) Serial.println("Memory OK");
    else Serial.println("Memory FAIL");
  }
}

void setup() {
  for (int i = 0; i < 24; i++)
    pinMode(i, INPUT);
  setupDram();
  Serial.begin(230400);
  delay(1000);
  inputString.reserve(200);
  cmd.reserve(20);
  param.reserve(20);
  Serial << "---------------------------------------------------------------------------------------\n";
  Serial << "Niteto ChipChecker xx464 " << Version << " " << __DATE__ << " - Copyright (C) 2017 Dominik Tonn (nick@niteto.de)\n";
  Serial << "released under GNU General Public License, for details see http://www.gnu.org/licenses/\n";
  Serial << "---------------------------------------------------------------------------------------\n";
  Serial << "data - tests databus for stuck lines\n";
  Serial << "adr  - tests adressbusfor stuck lines\n";
  Serial << "mem  - tests full memory range\n";
  Serial << "full - databus, adressbus and full memory test\n";
  Serial << "time - time the teensy took for a full refresh cycle\n";
  Serial << "help - lists available commands\n";
  delay(2000);
}

void loop() {
  if (Serial.available() != 0) {
    getCommand();
    doCommand();
  }
  zeit = micros();
  for (int i = 0; i < 256; i++)
    refresh(i);
  tRefresh = micros() - zeit;
}
