/*
   ADF-Copy/Drive - Reads and writes Amiga floppy disks
   Copyright (C) 2016-2019 Dominik Tonn (nick@niteto.de)

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
   encodeMFM is partly based upon floppySectorMfmEncode from FLOPPY.C which is part of the Win Fellow Project
   Authors: Petter Schau, Torsten Enderling (carfesh@gmx.net)
   Copyright (C) 1991, 1992, 1996 Free Software Foundation, Inc
   released under the GNU General Public License
*/

/*
   decodesMFM is partly based upon DecodeSectorData and DecodeLongword functions from AFR.C,
   written by Marco Veneri Copyright (C) 1997 and released as public domain
*/

/*
   MTP is based on MTP.h
   MTP.h - Teensy MTP Responder library
   Copyright (C) 2017 Fredrik Hubinette <hubbe@hubbe.net>
   With updates from MichaelMC, YoongHM
   see MTPADF.h for details
*/

/*
   This program uses a modified version of ADFLib
   http://lclevy.free.fr/adflib/

   ADF Library. (C) 1997-2002 Laurent Clevy
   ADFLib is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   ADFLib is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Foobar; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#define Version "v1.010"
#define numVersion 1010
#include "adfdrive_defines.h" //moved many defines

#ifdef __cplusplus
extern "C" {
#endif
/*
   Setup for the Watchdog Timer to 8 seconds
*/
void startup_early_hook() {
  WDOG_TOVALL = 8000;
  WDOG_TOVALH = 0;
  WDOG_PRESC = 0;
  WDOG_STCTRLH = (WDOG_STCTRLH_ALLOWUPDATE | WDOG_STCTRLH_WDOGEN);
}
#ifdef __cplusplus
}
#endif

//defines for the Teensy 3.2 board
#ifdef __MK20DX256__
#define __teensy
#define vTaskDelay delay
#define BITBAND_ADDR(addr, bit) (((uint32_t)&(addr) - 0x20000000) * 32 + (bit) * 4 + 0x22000000)
int _index, _drivesel, _motor, _dir, _step, _writedata, _writeen, _track0, _wprot, _readdata, _side, _diskChange;
//int _dens;
uint32_t FTChannelValue, FTStatusControlRegister, FTPinMuxPort;
IntervalTimer writeTimer;
IntervalTimer errorTimer;
byte * stream;
int * streamBitband;
#include "FreeStack.h"
#include <EEPROM.h>
#endif //Teensy

//defines for the ESP32 experimental board
#ifdef HAL_ESP32_HAL_H_
#define __esp
#define Version "ESP v0.100"
#define digitalWriteFast digitalWrite
#include "Esp.h"
#include "esp32-hal-timer.h"
#include "soc/timer_group_struct.h"
#include "rom/ets_sys.h"
#include "esp_intr.h"
#include "driver/periph_ctrl.h"
#include "driver/timer.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_int_wdt.h"
hw_timer_t *myTimer = NULL;
hw_timer_t *writeTimer = NULL;
uint32_t espWriteTimer = 160;
#define myTimerNum 3
#define myTimerDiv 4
#define TIM_CNT (*(volatile uint32_t *)TIMG_T1LO_REG(1))
#define TIM_UPDATE (*(volatile uint32_t *)TIMG_T1UPDATE_REG(1))
#define TIM_LOADLO (*(volatile uint32_t *)TIMG_T1LOADLO_REG(1))
#define TIM_LOADHI (*(volatile uint32_t *)TIMG_T1LOADHI_REG(1))
#define TIM_LOADREG (*(volatile uint32_t *)TIMG_T1LOAD_REG(1))
uint8_t stream[streamSizeHD + 10];
gpio_num_t _index, _drivesel, _motor, _dir, _step, _writedata, _writeen, _track0, _wprot, _readdata, _side, _diskChange;
#define __espwifi
#endif //ESP32

//#include "Streaming.h"
#include "TimeLib.h"
#include "adf_nativ.h"

//defines for MTPMode
boolean mtpOn = true;
boolean preErase = false;
#include "MTPADF.h"
MTPStorage_SD storage;
MTPD mtpd(&storage);
uint32_t imageHandle = 0xADFF17E;
uint32_t queuedEvent[2];

boolean badDisk;
boolean ADFisOpen = false;
#include "SdFat.h"
SdFat SD;
bool noSD = true;

//initialize default for DD Disks
byte low2 = tDDlow2;
byte high2 = tDDhigh2;
byte high3 = tDDhigh3;
byte high4 = tDDhigh4;
byte sectors = 11;

uint16_t timerMode = timerModeDD;
uint16_t filterSetting = filterSettingDD;
uint32_t writeSize = writeSizeDD;
uint32_t streamLen = streamSizeDD;
volatile boolean recordOn = false;

uint16_t hist[256];
byte weakTracks[168];
byte trackLog[168];
volatile uint32_t bitCount;
volatile uint32_t sectorCnt;
int32_t errorD, errorH;
int32_t errors;
int8_t extError = 0;
uint16_t failureCode = 0;
boolean fail = false;

int currentTrack = -1;
int logTrack = -1;
String inputString = "";
volatile boolean motor = false;
String cmd, param;
uint16_t retries = sdRetries_;
float transitionTime = transTimeDD;
int trackTemp;
int hDet, hDet2;
int densMode = DD;
#ifdef hdmode
boolean getDensity = true;
#else
boolean getDensity = false;
#endif
boolean wProt = true;
boolean noDisk = true;
volatile boolean dchgActive = true;
uint32_t diskTick = 0;
String volumeName (30, 32);
uint32_t motorTime = 0;
#ifdef errLed
uint16_t blinkCode = 0b0011001100110011;
#endif

struct tfloppyPos
{
  byte track;
  byte side;
  byte dir;
};

struct tfloppyPos floppyPos;

struct SectorTable
{
  unsigned long bytePos;
  byte sector;
};
struct SectorTable sectorTable[25];

struct Sector
{
  byte format_type;       //0
  byte track;             //1
  byte sector;            //2
  byte toGap;             //3
  byte os_recovery[16];   //4
  unsigned long header_chksum;  //20
  unsigned long data_chksum;    //24
  byte data[512];         //28
};

struct rawSector
{
  byte sector[540];
};

struct rawSector track[23];
byte tempSector[540];

uint32_t zeit;

uint32_t writePtr;
uint32_t writeBitCnt;
volatile byte dataByte;
volatile boolean writeActive;

// Rootblock (512 bytes) sector 880 for a DD disk, 1760 for a HD disk
struct rootBlock
{
  long  type;         // block primary type = T_HEADER (value 2)
  long  header_key;   // unused in rootblock (value 0)
  long  high_seq;     // unused (value 0)
  long  ht_size;      // Hash table size in long (value 0x48)
  long  first_data;   // unused (value 0)
  long  chksum;       // sum to check block integrity
  long  ht[72];       // hash table (entry block number)
  long  bm_flag;      // bitmap flag, -1 means VALID
  long  bm_pages[25]; // bitmap blocks pointers (first at 0)
  long  bm_ext[4];    // first bitmap extension block (Hard disks only)
  byte  name_len;     // disk name length
  char  diskname[30]; // disk name
  byte  filler[9];
  long  days;         // last access date : days since 1 jan 1978
  long  mins;         // minutes past midnight
  long  ticks;        // ticks (1/50 sec) past last minute
  long  c_days;       // creation date
  long  c_mins;
  long  c_ticks;
  long  next_hash;    // unused (value = 0)
  long  parent_dir;   // unused (value = 0)
  long  extension;    // FFS: first directory cache block, 0 otherwise
  long  sec_type;     // block secondary type = ST_ROOT (value 1)
};

// Bitmap block (512 bytes), often rootblock+1
struct bitmapBlock
{
  long  checksum; //  special algorithm
  long  map[127];
};

/*
   Feeds the Watchdog
*/
void watchdogReset() {
  static elapsedMillis watchdogTimer;

  if (watchdogTimer > 5) {
    watchdogTimer = 0;

    noInterrupts();
    WDOG_REFRESH = 0xA602;
    WDOG_REFRESH = 0xB480;
    interrupts();
  }
}

/*
   reads command from the serial port
*/
void getCommand()
{
  inputString = "";
  while (Serial.available())
  {
    char inChar = (char)Serial.read();
    inputString += inChar;
    if (inChar == '\n') {
      break;
    }
  }
}

/*
   execute command
*/
void doCommand()
{
  inputString.replace((char)10, (char)0);
  inputString.replace((char)13, (char)0);
  cmd = inputString;
  param = inputString;
  if (inputString.indexOf(" ") > 0) {
    cmd.remove(inputString.indexOf(" "), inputString.length());
    param.remove(0, inputString.indexOf(" "));
  } else {
    cmd = inputString;
    param = "";
  }
  cmd.trim();
  param.trim();
  //    Serial.printf("cmd: '%s' param: '%s'\n",cmd.c_str(),param.c_str());
  //    Serial.println(cmd);
  //    Serial.println(param);
  if (cmd == "help") {
    Serial.print("-----------------------------------------------------------------------------\n");
    Serial.printf("ADF-Drive %s - Copyright (C) 2016-2019 Dominik Tonn (nick@niteto.de)\n", Version);
    Serial.print("-----------------------------------------------------------------------------\n");
    Serial.print("read <n> - read logical track #n\n");
    Serial.print("write <n> - write logical track #n\n");
    Serial.print("erase <n> - erase logical track #n\n");
    Serial.print("testwrite <n> - write logical track #n filled with 0-255\n");
    Serial.print("print - prints amiga track with header\n");
    Serial.print("error - returns encoded error in ascii\n");
    Serial.print("exterr - returns a detailed error message in ascii\n");
    Serial.print("weak - returns retry number for last read in binary format\n");
    Serial.print("get <n> - reads track #n silent\n");
    Serial.print("put <n> - writes track #n silent\n");
    Serial.print("download - download track to pc without headers in binary\n");
    Serial.print("upload - upload track to mcu without headers in binary\n");
    Serial.print("init - goto track 0\n");
    Serial.print("hist - prints histogram of track in ascii\n");
    Serial.print("flux - returns histogram of track in binary\n");
    Serial.print("index - prints index signal timing in ascii\n");
    Serial.print("name - reads track 80 an returns disklabel in ascii\n");
    Serial.print("dskcng - returns disk change signal in binary\n");
    Serial.print("dens - returns density type of inserted disk in ascii\n");
    Serial.print("info - prints state of various floppy signals in ascii\n");
    Serial.print("enc - encodes data track into mfm\n");
    Serial.print("dec - decodes raw mfm into data track\n");
    Serial.print("log - prints logical track / tracknumber extracted from sectors\n");
    Serial.print("showsettings - shows current floppy access timings\n");
    Serial.print("getsettings - gets current floppy access timings\n");
    Serial.print("setsettings - sets timing values for floppy access\n");
    Serial.print("resetsettings - resets all timing values to default\n");
    Serial.print("storesettings - stores current values in eeprom\n");
    Serial.print("restoresettings - restores timing values from eeprom\n");
    Serial.print("diskinfo - show info about the current disk\n");
    Serial.print("bitmap - dumps rootblockbitmap in binary\n");
  }
  if (cmd == "setsettings") {
    sscanf(param.c_str(), "%lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu",
           &settings.motorSpinupDelay, &settings.motorSpindownDelay, &settings.driveSelectDelay,
           &settings.driveDeselectDelay, &settings.setDirDelay, &settings.setSideDelay,
           &settings.stepDelay, &settings.stepSettleDelay, &settings.gotoTrackSettle,
           &settings.sdRetries, &settings.hdRetries, &settings.config,
           &settings.reserved3, &settings.reserved4, &settings.reserved5,
           &settings.reserved6, &settings.reserved7);
    Serial.println("OK");
  }
  if (cmd == "getsettings") {
    Serial.printf("%lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu\n",
                  settings.motorSpinupDelay, settings.motorSpindownDelay, settings.driveSelectDelay,
                  settings.driveDeselectDelay, settings.setDirDelay, settings.setSideDelay,
                  settings.stepDelay, settings.stepSettleDelay, settings.gotoTrackSettle,
                  settings.sdRetries, settings.hdRetries, settings.config,
                  settings.reserved3, settings.reserved4, settings.reserved5,
                  settings.reserved6, settings.reserved7);
    Serial.println("OK");
  }
  if (cmd == "showsettings") {
    printTimings(settings);
    Serial.print("MTP is ");
    if (mtpOn) Serial.println("ON");
    else Serial.println("OFF");
    Serial.println("OK");
  }
  if (cmd == "resetsettings") {
    setDefaultTimings();
    Serial.println("OK");
  }
  if (cmd == "cleareeprom") {
    for (int i = 0 ; i < 80 ; i++) {
      EEPROM.write(i, 0xff);
    }
    Serial.println("OK");
  }
  if (cmd == "storesettings") {
    storeTimings();
    Serial.println("OK");
  }
  if (cmd == "restoresettings") {
    if (restoreTimings() == 0)
      Serial.println("OK");
    else
      Serial.println("Checksum Error");
  }
  if (cmd == "read") {
    Serial.printf("Reading Track %d\n", param.toInt());
    doRead(param.toInt(), false);
    Serial.printf("Track: %d Side: %d Dir: %d CurrentTrack: %d LogicalTrack: %d\n",
                  floppyPos.track, floppyPos.side, floppyPos.dir, currentTrack, logTrack);
  }

  if (cmd == "read81") {
    Serial.printf("Reading Track %d\n", param.toInt());
    doRead(param.toInt(), true);
    Serial.printf("Track: %d Side: %d Dir: %d CurrentTrack: %d LogicalTrack: %d\n",
                  floppyPos.track, floppyPos.side, floppyPos.dir, currentTrack, logTrack);
  }

  if (cmd == "write") {
    busy(true);
    Serial.printf("Writing Track %d\n", param.toInt());
    gotoLogicTrack(param.toInt());
    errors = writeTrack(preErase);
    if (errors != -1) {
      Serial.println("OK");
    } else {
      Serial.println("Write failed!");
    }
  }

  if (cmd == "erase") {
    busy(true);
    float tTrans = transTimeDD;
    int temp = 0;
    int tTrack = 0;
    sscanf(param.c_str(), "%d %d", &tTrack, &temp);
    tTrans = (float)temp / 100;
//    Serial.printf("Erasing Track: %d tTime: ", tTrack);
//    Serial.println(tTrans);
    gotoLogicTrack(tTrack);
    errors = eraseTrack(tTrans);
//    Serial.println(writeBitCnt);
    if (errors != -1) {
      Serial.println("OK");
    } else {
      Serial.println("Erase failed!");
    }
  }
  if (cmd == "edisk") {
    busy(true);
    Serial.println("Erasing Disk.");
    errors = eraseDisk(4.0);
    Serial.println(writeBitCnt);
    if (errors != -1) {
      Serial.println("OK");
    } else {
      Serial.println("Erase failed!");
    }
  }
  if (cmd == "ttest") {
    Serial.printf("Testing Track %d\n", param.toInt());
    gotoLogicTrack(param.toInt());
    for (int i = 200; i <= 1000; i = i + 20)
    {
      eraseTrack((float)i / 100);
      indexTimer();
      Serial.print((float)i / 100);
      Serial.printf(" write: %6d read: %6d diff: %5d\n", writeBitCnt, bitCount, writeBitCnt - bitCount);
    }

  }
  if (cmd == "probe") {
    gotoLogicTrack(param.toInt());
    Serial.println(probeTrack(false));
  }
  if (cmd == "preerase") {
    preErase = true;
    Serial.println("preerase on");
  }
  if (cmd == "noerase") {
    preErase = false;
    Serial.println("preerase off");
  }
  if (cmd == "put") {
    busy(true);
    trackTemp = param.toInt();
    floppyTrackMfmEncode(trackTemp, (byte*)track, stream);
    gotoLogicTrack(trackTemp);
    errors = writeTrack(preErase);
    if (errors == 0) {
      Serial.println("OK");
    } else {
      Serial.println("ERROR");
    }
  }

  if (cmd == "print") {
    printTrack();
    Serial.println("OK");
  }

  if (cmd == "raw") {
    for (uint32_t i = 0; i < streamLen; i++) {
      if ((i % 64 == 0) && (stream[i] != 0)) {
        Serial.println();
      }
      if (stream[i] != 0) {
        Serial.print(stream[i], HEX);
      }
    }
    Serial.println("\nOK");
  }

  if (cmd == "dump") {
    byte tByte = 0;
    for (uint32_t i = 0; i < streamLen; i = i + 2) {
      if ((i % 128 == 0) && (tByte != 0)) {
        Serial.println();
      }
      tByte = (stream[i] & 0x55) | ((stream[i + 1] & 0x55) << 1);
      if (tByte != 0) {
        Serial.print(byte2char(tByte));
      }
    }
    Serial.println("\nOK");
  }

  if (cmd == "error") {
    Serial.print(errorD | errorH);
    Serial.println();
  }

  if (cmd == "exterr") {
    switch (extError) {
      case NO_ERR:
        Serial.println("OK\0");
        break;
      case WPROT_ERR:
        Serial.println("Disk is write-protected");
        break;
      case TIMEOUT_ERR:
        Serial.println("Timeout in read operation");
        break;
      case SEEK_ERR:
        Serial.println("Seek error");
        break;
      case NUMSECTOR_ERR:
        Serial.println("Incorrect number of sectors on track");
        break;
      case TRACK0_ERR:
        Serial.println("Seek Error Track 0");
        break;
      case SECBOUNDS_ERR:
        Serial.println("Sector out of bounds");
        break;
      case HEADERCHKSUM_ERR:
        Serial.println("Header bad checksum");
        break;
      case DATACHKSUM_ERR:
        Serial.println("Data bad checksum");
        break;
      case UPLOAD_ERR:
        Serial.println("Track upload error");
        break;
      default:
        Serial.println(extError);

    }
  }

  if (cmd == "weak") {
    Serial.write(weakTracks[logTrack]);
  }

  if (cmd == "get") {
    gotoLogicTrack(param.toInt());
    readTrack(true, false);
  }

  if (cmd == "verify") {
    gotoLogicTrack(param.toInt());
    readTrack(false, true);
  }

  if (cmd == "goto") {
    gotoLogicTrack(param.toInt());
  }

  if (cmd == "download") {
    downloadTrack();
  }

  if (cmd == "upload") {
    busy(true);
    uploadTrack();
    Serial.println("OK");
  }

  if (cmd == "tw") {
    busy(true);
    motorOn();
    for (int i = 0; i < sectors; i++) {

      fillSector(i, true);
    }
    trackTemp = param.toInt();
    floppyTrackMfmEncode(trackTemp, (byte*)track, stream);
    gotoLogicTrack(trackTemp);
    writeTrack(false);
    retries = 6;
    delay(500);
    if (readTrack(false, false) != -1) {
      Serial.printf("Sectors found: %d  Errors found: ", sectorCnt);
      Serial.print(errorD | errorH);
      Serial.println();
      Serial.println("OK");
    } else {
      Serial.println("Read failed!");
    }
    retries = settings.sdRetries;
  }

  if (cmd == "hist") {
    analyseHist(false);
    printHist(128);
    Serial.println("OK");
  }

  if (cmd == "flux") {
    analyseHist(true);
    printFlux();
  }

  if (cmd == "index") {
    Serial.printf("%d ms, transitions: %d\n", indexTimer(), bitCount);
    Serial.println("OK");
  }

#ifdef hdmode
  if (cmd == "getmode") {
    getDensity = true;
    motorOn();
    if (densMode == HD) {
      Serial.println("HD");
    } else {
      Serial.println("DD");
    }
  }
#endif

  if (cmd == "name") {
    volumeName = getName();
    Serial.println(volumeName);
  }

  if (cmd == "diskinfo") {
    diskInfo();
    Serial.println("OK");
  }

  if (cmd == "di") {
    diskInfoOld();
    Serial.println("OK");
  }

  if (cmd == "bitmap") {
    dumpBitmap();
    Serial.println("OK");
  }

  if (cmd == "dskcng") {
    Serial.println(diskChange());
  }

#ifdef hdmode
  if (cmd == "setmode") {
    if (param == "hd") {
      getDensity = false;
      setMode(HD);
      Serial.println("HD");
    } else {
      getDensity = false;
      setMode(DD);
      Serial.println("DD");
    }
  }
#endif

  if (cmd == "ver") {
    Serial.printf("ADF-Drive/Copy Controller Firmware %s", Version);
    if (hDet == 0) Serial.println(" Breadboard");
    if (hDet == 1) Serial.println(" V3/4 PCB");
    if (hDet == 2) Serial.println(" ESP32");
    if (hDet == -1) Serial.println(" Unknown Hardware");
  }

  if (cmd == "enc") {
    busy(true);
    trackTemp = param.toInt();
    Serial.printf("Encoding Track %d to MFM\n", trackTemp);
    uint32_t tZeit = micros();
    floppyTrackMfmEncode(trackTemp, (byte*)track, stream);
    tZeit = micros() - tZeit;
    Serial.printf("done in %d micros\n", tZeit);
  }

  if (cmd == "dec") {
    busy(true);
    Serial.println("Decode MFM");
    decodeTrack(false, false, false);
    Serial.println("done");
  }

  if (cmd == "log") {
    Serial.println("Tracking Info");
    for (int i = 0; i < 160; i++) {
      if (i != trackLog[i]) {
        Serial.printf("Track: %d Real Track: %d\n", i, trackLog[i]);
      }
    }
  }

  if (cmd == "mfm") {
    Serial.println(mfmByte(param.toInt(), 0), HEX);
  }

  if (mtpOn) {
    if (cmd == "event") {
      int x, y = 0;
      sscanf(param.c_str(), "%x %d", &x, &y);
      Serial.printf("x: %#.4x y: %d\n", x, y);
      mtpd.sendEvent(x, y);
    }
    if (cmd == "busy") {
      busy(true);
    }
    if (cmd == "nobusy") {
      busy(false);
    }
  }
  if (cmd == "nomtp") {
    mtpOn = false;
  }
  if (cmd == "mtp") {
    if ((settings.config & 0x01) == 1) mtpOn = true;
    else mtpOn = false;
  }

  if (cmd == "retry") {
    Serial.printf("Old retry value: %d\n", retries);
    retries = param.toInt();
    Serial.printf("New retry value: %d\n", retries);
  }

  if (cmd == "ana") {
    busy(true);
    analyseTrack(false, param.toInt());
    printHist(-1);
  }

  if (cmd == "format") {
    busy(true);
    uint32_t tZeit = millis();
    int fErr = 0;
    if (param == "quick")
      fErr = formatDisk(true, true);
    else
      fErr = formatDisk(false, false);
    tZeit = millis() - tZeit;
    if (fErr == 0) Serial.printf("Format done in %d seconds\n", tZeit / 1000);
    else Serial.printf("Format failed in %d seconds\n", tZeit / 1000);
  }

  if (cmd == "getdate") {
    struct DateTime timeTemp;
    timeTemp = adfGiveCurrentTime();
    Serial.printf("adfTime: %02d:%02d:%02d %02d.%02d.%04d\n",
                  timeTemp.hour,
                  timeTemp.min,
                  timeTemp.sec,
                  timeTemp.day,
                  timeTemp.mon,
                  timeTemp.year + 1900);
  }

  if (cmd == "setdate") {
    struct DateTime timeTemp;
    timeTemp = adfGiveCurrentTime();
    Serial.printf("adfTime: %02d:%02d:%02d %02d.%02d.%04d\n",
                  timeTemp.hour,
                  timeTemp.min,
                  timeTemp.sec,
                  timeTemp.day,
                  timeTemp.mon,
                  timeTemp.year + 1900);
  }

  if (cmd == "init") {
    busy(true);
#ifdef hdmode
    getDensity = true;
#endif
    currentTrack = -1;
    if (diskChange() == 1) {
      motorOn();
      seek0();
    }
    errors = 0;
  }

  if (cmd == "rst") {
    delay(1000);
    CPU_RESTART
  }
  if (cmd == "info") {
    printStatus();
    Serial.print("Pin0: ");
    Serial.println(digitalRead(0));
#ifdef __teensy
    Serial.printf("FreeStack: %d\n", FreeStack());
#endif
#ifdef __esp
    Serial.print("Free Heap: ");
    Serial.println(ESP.getFreeHeap());
    Serial.print("SDK Version: ");
    Serial.println(ESP.getSdkVersion());
    Serial.print("Tasks: ");
    Serial.println(uxTaskGetNumberOfTasks());

    printTimerConfig(*(volatile uint32_t *)TIMG_T0CONFIG_REG(0));
    printTimerConfig(*(volatile uint32_t *)TIMG_T0CONFIG_REG(1));
    printTimerConfig(*(volatile uint32_t *)TIMG_T1CONFIG_REG(0));
    printTimerConfig(*(volatile uint32_t *)TIMG_T1CONFIG_REG(1));
    /*
        Serial << "ccount: " << (xthal_get_ccount()) << "\n";
        Serial << "ccompare0: " << (xthal_get_ccompare(0)) << "ccompare1: " << (xthal_get_ccompare(1)) << "ccompare2: " << (xthal_get_ccompare(2)) << "\n";
        Serial << "GetIntEnable: " << xthal_get_intenable() << "\n";
        xthal_set_intclear(XCHAL_TIMER0_INTERRUPT);
        taskDISABLE_INTERRUPTS();
        xthal_set_intenable(0);
        Serial << "GetIntEnable: " << xthal_get_intenable() << "\n";
        xthal_set_intset(XCHAL_TIMER0_INTERRUPT);
        taskENABLE_INTERRUPTS();
        xthal_set_intenable(5220);
        Serial << "GetIntEnable: " << xthal_get_intenable() << "\n";
        timerStart(myTimer);
        for (int i = 0; i < 20; i++) {
          TIM_UPDATE = 1;
          volatile uint32_t t1 = TIM_CNT;
          volatile uint32_t t2 = TIM_CNT;
          Serial << t1 << " " << t2 << "\n";
        }
        timerStop(myTimer);
    */
    /*
      uint32_t tLast = 0;
      uint32_t low = 0;
       for (int i = 0; i < 10; i++) {
         Serial.println(TIM_CNT);
         TIM_UPDATE =1;
         low = TIM_CNT;
         Serial.print("Timer: ");
         Serial.println(low);
         Serial.print("Diff: ");
         Serial.println(low-tLast);
         tLast=low;
         vTaskDelay(100);
       }
    */
    //    Serial.println(hw_timer[0]);

#endif
  }
}

#ifdef __esp
/*
   Debug function for ESP32
*/
void printTimerConfig(uint32_t tConfig)
{
  tConfig = tConfig >> 10;
  Serial.print("AlarmEn: ");
  Serial.print(tConfig & 0B1);
  tConfig = tConfig >> 1;
  Serial.print(" level_int_en: ");
  Serial.print(tConfig & 0B1);
  tConfig = tConfig >> 1;
  Serial.print(" edge_int_en: ");
  Serial.print(tConfig & 0B1);
  tConfig = tConfig >> 1;
  Serial.print(" divider: ");
  Serial.print(tConfig & 0xFFFF);
  tConfig = tConfig >> 16;
  Serial.print(" autoreload: ");
  Serial.print(tConfig & 0B1);
  tConfig = tConfig >> 1;
  Serial.print(" increase: ");
  Serial.print(tConfig & 0B1);
  tConfig = tConfig >> 1;
  Serial.print(" enabled: ");
  Serial.println(tConfig & 0B1);
}
#endif //ESP

/*
   Setup Teensy, Hardware and other stuff
*/
void setup() {
#ifdef __esp
#define espadf 2
  hDet =  espadf; //hardwareVersion();
  registerSetup(espadf);
#else // Teensy
  pinMode(0, INPUT_PULLUP);
  if ((restoreTimings() == -1) || (digitalRead(0) == 0)) storeDefaultTimings();
  hDet = hardwareVersion();
  registerSetup(hDet);
#endif
#ifdef errLed
  startErrorLed();
#endif
  while (hDet == -1) { // No drive detected? Blink and loop
    failureCode = hardError;
    blinkCode = failureCode;
    watchdogReset();
    delay(1000);
    hDet = hardwareVersion(); // check if something changed
    if (hDet != -1) {
      registerSetup(hDet);
      failureCode = 0;
      blinkCode = failureCode;
    }
  }
  if (hDet != -1) {
    pinMode(_index, INPUT_PULLUP);
    pinMode(_motor, OUTPUT);
    digitalWrite(_motor, HIGH);
    pinMode(_drivesel, OUTPUT);
    digitalWrite(_drivesel, LOW);
    pinMode(_dir, OUTPUT);
    digitalWrite(_dir, HIGH);
    pinMode(_step, OUTPUT);
    digitalWrite(_step, HIGH);
#ifdef __esp
    pinMode(_writedata, OUTPUT);
#else
    pinModeFast(_writedata, OUTPUT);
#endif
    digitalWrite(_writedata, HIGH);
    pinMode(_writeen, OUTPUT);
    digitalWrite(_writeen, HIGH);
    pinMode(_track0, INPUT_PULLUP);
    pinMode(_wprot, INPUT_PULLUP);
    pinMode(_side, OUTPUT);
    digitalWrite(_side, HIGH);
    pinMode(_diskChange, INPUT_PULLUP);
  }
  Serial.begin(230400);
  delay(100);
  inputString.reserve(200);
  extError = NO_ERR;
  cmd.reserve(20);
  param.reserve(20);
  Serial.println("---------------------------------------------------------------------------------------");
  Serial.printf("ADF-Drive/Copy %s %s - Copyright (C) 2016-2019 Dominik Tonn (nick@niteto.de)\n", Version, __DATE__);
  Serial.printf("released under GNU General Public License, for details see http://www.gnu.org/licenses/)\n");
  Serial.printf("---------------------------------------------------------------------------------------\n");
  Serial.printf("type help for commands\n");
#ifdef __teensy
  byte * tempMem;
  tempMem = (byte*) malloc(1);

  uint32_t foo = 0x20000000 - (long)tempMem;
  free(tempMem);
  tempMem = (byte*) malloc(foo);

  // workaround to prevent the compiler from optimizing away the malloc(foo)
  // for aligning the stream memory to bitband memory space
  // you might get a warning that preventCompilerOptimize is never used.
  volatile uint32_t preventCompilerOptimize __attribute__((unused));
  preventCompilerOptimize = (long)&tempMem[foo];
  // end of workaround

  stream = (byte*) malloc(streamSizeHD + 10);
  streamBitband = (int*) BITBAND_ADDR(*stream, 0);
#ifdef debug
  printPtr(tempMem, "tempMem");
  printPtr(stream, "stream");
#endif

  if (stream == 0) {
    Serial.println("Out of memory");
    failureCode = memError;
    blinkCode = memError;

    while (1) { // No memory for Bitband -> blink forever
      watchdogReset();
      delay(1000);
    }
  }
  free(tempMem);
#endif

  initRead();
  floppyPos.dir = 0;
  floppyPos.side = 0;
  floppyPos.track = 0;
#ifdef __esp
  myTimer = timerBegin(myTimerNum, myTimerDiv, true); // 1µS = 20 counts
#endif
#ifdef __espwifi
  startServerTask();
#endif
  if (mtpOn) {
    queuedEvent[0] = 0;
  }
  noSD = true;
  //  noSD = !SD.begin(10, SPI_QUARTER_SPEED);
#ifdef debug
  if (noSD) {
    Serial.println("SD not found");
  } else {
    Serial.println("SD found");
  }
#endif
  attachInterrupt(_diskChange, diskChangeIRQ, FALLING);
}

#ifdef debug
void printPtr(byte * dest, String pName)
{
  Serial.printf("%s: %xl -> %xl\n", pName,(long)dest,(long)*dest);
}
#endif

/*
   Main loop for doing everything
   Check Serial, handle MTP requests, detect inserted Disk
*/
void loop() {
  if (Serial.available() != 0) {
    getCommand();
    doCommand();
  }
  if ((millis() - motorTime) > 5000) {
    if (motor) {
      if (ADFisOpen) flushToDisk();
      motorOff();
    }
  }
#ifdef __esp
  vTaskDelay(10); //yield some time for wifi task
#endif
  if (!motor) {
    if ((millis() - diskTick) > 500) {
      driveSelect();
      driveDeselect();
      if (noDisk) {
        if (diskChange() == 1) {
          failureCode = 0;
          indexTimer();
          if (bitCount < 160000) {
            badDisk = false;
            if (mtpOn) {
              mtpd.sendEvent(MTP_EVENT_STORE_ADDED, FDid);
              mtpd.sendEvent(MTP_EVENT_STORE_ADDED, FSid);
            }
          } else {
            badDisk = true;
            if (mtpOn) {
              mtpd.sendEvent(MTP_EVENT_STORE_ADDED, FDid);
            }
          }
          dchgActive = true;
        }
        //     } else {
      }
      diskTick = millis();
    }
  }
  if (mtpOn) {
    mtpd.loop();
    if (fail) {
#ifdef errLed
      blinkCode = failureCode;
#endif
      fail = false;
    }
    if (queuedEvent[0] != 0)
    {
      mtpd.sendEvent(queuedEvent[0], queuedEvent[1]);
      queuedEvent[0] = 0;
    }
  }
#ifdef __teensy
  watchdogReset();
#endif
}
