/*
   This Module handles all the low level access to the drive, reading, writing,
   moving the head, i think you get it
*/
// bswap define für ESP32
#ifdef __esp
/* Swap bytes in 32 bit value.  */
#ifdef __GNUC__
# define __bswap32(x) \
  (__extension__                    \
  ({ unsigned int __bsx = (x);               \
    ((((__bsx) & 0xff000000) >> 24) | (((__bsx) & 0x00ff0000) >>  8) |    \
     (((__bsx) & 0x0000ff00) <<  8) | (((__bsx) & 0x000000ff) << 24)); }))
#else
static INLINE unsigned int
__bswap32 (unsigned int __bsx)
{
  return ((((__bsx) & 0xff000000) >> 24) | (((__bsx) & 0x00ff0000) >>  8) |
          (((__bsx) & 0x0000ff00) <<  8) | (((__bsx) & 0x000000ff) << 24));
}
#endif
#endif

#ifdef __teensy
/*
   sets teensy pin to fastmode if OUTPUT
   slightly modified pinMode function from
   \hardware\teensy\avr\cores\teensy3\pins_teensy.c
*/
void pinModeFast(uint8_t pin, uint8_t mode)
{
  volatile uint32_t *config;

  if (pin >= CORE_NUM_DIGITAL) return;
  config = portConfigRegister(pin);

  if (mode == OUTPUT || mode == OUTPUT_OPENDRAIN) {
#ifdef KINETISK
    *portModeRegister(pin) = 1;
#else
    *portModeRegister(pin) |= digitalPinToBitMask(pin); // TODO: atomic
#endif
    *config = PORT_PCR_DSE | PORT_PCR_MUX(1);
    if (mode == OUTPUT_OPENDRAIN) {
      *config |= PORT_PCR_ODE;
    } else {
      *config &= ~PORT_PCR_ODE;
    }
  } else {
#ifdef KINETISK
    *portModeRegister(pin) = 0;
#else
    *portModeRegister(pin) &= ~digitalPinToBitMask(pin);
#endif
    if (mode == INPUT || mode == INPUT_PULLUP || mode == INPUT_PULLDOWN) {
      *config = PORT_PCR_MUX(1);
      if (mode == INPUT_PULLUP) {
        *config |= (PORT_PCR_PE | PORT_PCR_PS); // pullup
      } else if (mode == INPUT_PULLDOWN) {
        *config |= (PORT_PCR_PE); // pulldown
        *config &= ~(PORT_PCR_PS);
      }
    } else {
      *config = PORT_PCR_MUX(1) | PORT_PCR_PE | PORT_PCR_PS; // pullup
    }
  }
}
#endif

/*
   wait for index hole
*/
void waitForIndex()
{
  motorOn();
  while (digitalRead(_index) == 1);
  while (digitalRead(_index) == 0);
}

unsigned char reverse(unsigned char b) {
  b = (b & 0xF0) >> 4 | (b & 0x0F) << 4;
  b = (b & 0xCC) >> 2 | (b & 0x33) << 2;
  b = (b & 0xAA) >> 1 | (b & 0x55) << 1;
  return b;
}

#ifdef __teensy
/*
   interrupt routine for writing data to floppydrive
*/
void diskWrite()
{
  if (writeActive == false) return;
  digitalWriteFast(_writedata, !streamBitband[writePtr]);
  writePtr++;
  if (writePtr >= ((writeSize * 8) + 8)) {
    writeActive = false;
    digitalWriteFast(_writedata, HIGH);
  }
}

/*
   old diskwrite function without bitband memory i'll let this here
   in case to port this to another microcontroller
*/
void olddiskWrite()
{
  if (writeActive == false) return;
  digitalWriteFast(_writedata, !(dataByte >> 7));
  dataByte = dataByte << 1;
  writeBitCnt++;
  if (writeBitCnt == 8) {
    writePtr++;
    dataByte = stream[writePtr];
    writeBitCnt = 0;
  }
  if (writePtr > writeSize) {
    writeActive = false;
    digitalWriteFast(_writedata, HIGH);
  }
}

/*
   writes a track from buffer to the floppydrive
   return -1 if disk is write protected
*/
int writeTrack(boolean erase)
{
  if (erase) eraseTrack(transitionTime * 2);
  watchdogReset();
  motorOn();
  if (writeProtect()) {
    extError = WPROT_ERR;
    return -1;
  }
  for (uint32_t i = 0; i < writeSize; i++) {
    stream[i] = reverse(stream[i]);
  }
  writePtr = 0;
  writeBitCnt = 0;
  dataByte = stream[writePtr];
  writeActive = true;
  writeTimer.priority(0);
  delayMicroseconds(100);
  //waitForIndex();
  int zeit = millis();
  writeTimer.begin(diskWrite, transitionTime);
  digitalWriteFast(_writeen, LOW); // enable writegate after starting timer because first interrupt
  // occurs in about 2µs
  while (writeActive == true) {}
  writeTimer.end();
  zeit = millis() - zeit;
  // Serial << "Time taken: " << zeit << "ms\n";
  delayMicroseconds(2);
  digitalWriteFast(_writeen, HIGH);
  vTaskDelay(5);
  return 0;
}

/*
   interrupt routine for erasing data on floppydrive
*/
void writeErase()
{
  /*  static boolean writeBit = false;
    if (writeActive == false) return;
    digitalWriteFast(_writedata, writeBit);
    writeBit=!writeBit;
    writeBitCnt++;
  */
  if (writeActive == false) return;
  digitalWriteFast(_writedata, LOW);
  writeBitCnt++;
  digitalWriteFast(_writedata, HIGH);
}

/*
   writes 01 with a given frequency to the floppydrive
   return -1 if disk is write protected
*/
int eraseTrack(float transitionDelay)
{
  watchdogReset();
  motorOn();
  if (writeProtect()) {
    extError = WPROT_ERR;
    return -1;
  }
  writeBitCnt = 0;
  writeActive = true;
  writeTimer.priority(0);
  writeTimer.begin(writeErase, transitionDelay);
  digitalWriteFast(_writeen, LOW); //
  delay(205);
  writeTimer.end();
  delayMicroseconds(2);
  digitalWriteFast(_writeen, HIGH);
  vTaskDelay(5);
  return 0;
}

/*
   erases the complete disk as fast as possible
   return -1 if disk is write protected
*/
int eraseDisk(float transitionDelay)
{
  watchdogReset();
  motorOn();
  if (writeProtect()) {
    extError = WPROT_ERR;
    return -1;
  }
  gotoLogicTrack(0);
  setDir(1);
  writeBitCnt = 0;
  writeActive = true;
  writeTimer.priority(0);
  int zeit = millis();
  writeTimer.begin(writeErase, transitionDelay);
  for (int i = 0; i < 80; i++) {
    Serial.printf("%d\n", i * 2);
    digitalWriteFast(_writeen, LOW);
    delay(205);
    digitalWriteFast(_writeen, HIGH);
    watchdogReset();
    setSide(1);
    Serial.printf("%d\n", i * 2 + 1);
    digitalWriteFast(_writeen, LOW);
    delay(205);
    digitalWriteFast(_writeen, HIGH);
    watchdogReset();
    setSide(0);
    step1();
  }
  writeTimer.end();
  zeit = millis() - zeit;
  gotoLogicTrack(0);
  return 0;
}
#endif

/*
  tests if inserted media is a HD or DD disk
  1 = dd
  2 = hd
  0 = defect media
  -1 = write protected media
*/
int16_t probeTrack(boolean report)
{
  if (eraseTrack(2.00) == -1) return -1;
  indexTimer();
  int32_t diff = writeBitCnt - bitCount;
  if (report) Serial.printf(" write: %6d read: %6d diff: %5d\n", writeBitCnt, bitCount, diff);
  eraseTrack(4.00);
  indexTimer();
  if (bitCount < 45000) return 0;
  if (diff > 500) return 1;
  eraseTrack(2.00);
  return 2;
}

/*
   experimental functions for the ESP32 port
*/
#ifdef __esp

#define WRITE_CNT (*(volatile uint32_t *)TIMG_T1LO_REG(0))
#define WRITE_UPDATE (*(volatile uint32_t *)TIMG_T1UPDATE_REG(0))
#define WRITE_LOADLO (*(volatile uint32_t *)TIMG_T1LOADLO_REG(0))
#define WRITE_LOADHI (*(volatile uint32_t *)TIMG_T1LOADHI_REG(0))
#define WRITE_LOADREG (*(volatile uint32_t *)TIMG_T1LOAD_REG(0))
#define WRITE_SET (*(volatile uint32_t *)GPIO_OUT_W1TS_REG)
#define WRITE_CLEAR (*(volatile uint32_t *)GPIO_OUT_W1TC_REG)

#define WDT_PROTECT (*(volatile uint32_t *)TIMG_WDTWPROTECT_REG(1))
#define WDT_FEED (*(volatile uint32_t *)TIMG_WDTFEED_REG(1))

void feedWdt()
{
  WDT_PROTECT = TIMG_WDT_WKEY_VALUE;
  WDT_FEED = 0x1337FEED;
  WDT_PROTECT = 0xDEADFEED;
}
/*
   writes a track from buffer to the floppydrive
   return -1 if disk is write protected
*/
int writeTrack()
{
  uint32_t bitMask = 1 << _writedata;
  motorOn();
  if (writeProtect()) {
    extError = WPROT_ERR;
    return -1;
  }
  /*
    for (int i = 0; i < writeSize; i++) {
      stream[i] = reverse(stream[i]);
    }
  */
  writeActive = true;
  delayMicroseconds(100);
  //waitForIndex();
  int zeit = millis();
  digitalWrite(_writeen, LOW); // enable writegate after starting timer because first interrupt
  // occurs in about 2µs

  uint32_t bitCount = 0;
  writeTimer = timerBegin(1, espWriteTimer, true);
  WRITE_LOADLO = 0;
  WRITE_LOADHI = 0;
  timerStart(writeTimer);
  WRITE_UPDATE = 1;
  uint32_t writePtr = 0;
  byte writeBitCnt = 0;
  byte dataByte;
  dataByte = stream[writePtr];
  vTaskSuspendAll();
  taskDISABLE_INTERRUPTS();
  //  portMUX_TYPE foo = portMUX_INITIALIZER_UNLOCKED;
  //  taskENTER_CRITICAL(&foo);
  WRITE_LOADREG = 1;
  while (writePtr <= writeSize) {
    do
    {
      WRITE_UPDATE = 1;
    } while (WRITE_CNT == 0);
    bitCount++;
    feedWdt();
    if (dataByte < 0x80) {
      WRITE_SET = bitMask;
    } else {
      WRITE_CLEAR = bitMask;
    }

    dataByte = dataByte << 1;
    writeBitCnt++;

    if (writeBitCnt == 8) {
      writePtr++;
      dataByte = stream[writePtr];
      writeBitCnt = 0;
    }

    WRITE_LOADREG = 1;
    WRITE_UPDATE = 1;
  }
  //  taskEXIT_CRITICAL(&foo);
  taskENABLE_INTERRUPTS();
  xTaskResumeAll();
  WRITE_UPDATE = 1;
  timerStop(writeTimer);
  timerEnd(writeTimer);

  zeit = millis() - zeit;
  // Serial << "Time taken: " << zeit << "ms\n";
  delayMicroseconds(2);
  digitalWrite(_writeen, HIGH);
  vTaskDelay(5);
  return 0;
}
#endif

#ifdef __teensy
/*
   Initializes the Registers for the FlexTimer0 Module
*/
void setupFTM0() {
  // Input Filter waits for n cycles of stable input
  FTM0_FILTER = filterSetting;

  // Enable the FlexTimerModule and write registers
  // FAULTIE=0, FAULTM=00, CAPTEST=0, PWMSYNC=0, WPDIS=1, INIT=0, FTMEN=1
  FTM0_MODE = 0x05;

  // Initialize Timer registers
  FTM0_SC = 0x00;    // Diable Interrupts and Clocksource before initialization
  FTM0_CNT = 0x0000; // set counter value to 0
  FTM0_MOD = 0xFFFF; // set modulo to max value
  (*(volatile uint32_t *)FTStatusControlRegister) = 0x48;  // CHF=0  CHIE=1 (enable interrupt)
  // MSB=0  MSA=0 (Channel Mode Input Capture)
  // ELSB=1 ELSA=0 (Input Capture on falling edge)
  // DMA=0  DMA off

  // Enable FTM0 interrupt inside NVIC
  NVIC_SET_PRIORITY(IRQ_FTM0, 0);
  NVIC_ENABLE_IRQ(IRQ_FTM0);
  (*(volatile uint32_t *)FTPinMuxPort) = 0x403; // setup pin for Input Capture FTM0 in Pin Mux
}

// only modified by the ISR therefore not volatile;
uint8_t sample = 0;
int16_t sectLen = 0;
uint8_t bCnt = 0;
uint32_t readBuff;
uint32_t readPtr;

/*
   Interrupt Service Routine for FlexTimer0 Module
*/
extern "C" void ftm0_isr(void) {
  sample = (*(uint32_t *)FTChannelValue);
  // Reset count value
  FTM0_CNT = 0x0000;

  (*(volatile uint32_t *)FTStatusControlRegister) &= ~0x80; // clear channel event flag
  // skip too short / long samples, occur usually in the track gap
  bitCount++;
  if (sample > high4) {
    return;
  }
  if (sample < low2) {
    return;
  }
  // fills buffer according to transition length with 10, 100 or 1000 (4,6,8µs transition)
  readBuff = (readBuff << 2) | B10;
  bCnt += 2;
  if (sample > high2) {
    readBuff = readBuff << 1;
    bCnt++;
  }
  if (sample > high3) {
    readBuff = readBuff << 1;
    bCnt++;
  }
  if (bCnt >= 8) // do we have a complete byte?
  {
    stream[readPtr++] = readBuff >> (bCnt - 8); // store byte in streambuffer
    bCnt = bCnt - 8; // decrease bit count by 8
  }
  if (readBuff == 0xA4489448) { // look for magic word. usually 44894489, but detecting this way its
    // easier to byte align the received bitstream from floppy
    if (sectorCnt < sectors) { // as long we dont have x sectors store the sector start in a table
      sectorTable[sectorCnt].bytePos = readPtr - 7;
      sectorCnt++;
      bCnt = 4; // set bit count to 4 to align to byte
    }
  }
  hist[sample]++; // add sample to histogram
  if (readPtr > streamLen) // stop when buffer is full
  {
    recordOn = false;
    FTM0_SC = 0x00; // Timer off
  }
}
#endif

/*
   experimental functions for the ESP32 port
*/
#ifdef __esp
int IRAM_ATTR espRead()
{
  uint32_t readPtr = 0;
  uint32_t sample;
  uint8_t bCnt = 0;
  uint32_t readBuff = 0;
  //#define readMask 0x400000
  uint32_t readMask = 1 << _readdata;
  bitCount = 0;
  //  vTaskSuspendAll();
  taskDISABLE_INTERRUPTS();
  uint32_t intLevels = xthal_get_intenable();
  //xthal_set_intenable(0B0001010001100100);
  //xthal_set_intenable(0x1464);
  while (readPtr < streamLen) {
    //xthal_set_intenable(0x1420);
    while ((GPIO.in & readMask) != 0)
    {
      TIM_UPDATE = 1;
      if (TIM_CNT > 50000) {
        taskENABLE_INTERRUPTS();
        //        xTaskResumeAll();
        //xthal_set_intenable(intLevels);
        return -1;
      }
    }
    while ((GPIO.in & readMask) == 0);
    TIM_UPDATE = 1;
    //    xthal_set_intenable(0x1424);
    feedWdt();
    sample = TIM_CNT;
    TIM_LOADREG = 1;

    // skip too short / long samples, occur usually in the track gap
    bitCount++;
    if (sample > high4) {
      continue;
    }
    if (sample < low2) {
      continue;
    }
    hist[sample]++; // add sample to histogram

    // fills buffer according to transition length with 10, 100 or 1000 (4,6,8µs transition)
    readBuff = (readBuff << 2) | B10;
    bCnt += 2;
    if (sample > high2) {
      readBuff = readBuff << 1;
      bCnt++;
    }
    if (sample > high3) {
      readBuff = readBuff << 1;
      bCnt++;
    }
    if (bCnt >= 8) // do we have a complete byte?
    {
      stream[readPtr] = (byte)(readBuff >> (bCnt - 8)); // store byte in streambuffer
      bCnt = bCnt - 8; // decrease bit count by 8
      readPtr++; // adjust pointer to next byte in stream
    }

    if (readBuff == 0xA4489448) { // look for magic word. usually 44894489, but detecting this way its
      // easier to byte align the received bitstream from floppy
      if (sectorCnt < sectors) { // as long we dont have x sectors store the sector start in a table
        sectorTable[sectorCnt].bytePos = readPtr - 7;
        sectorCnt++;
        bCnt = 4; // set bit count to 4 to align to byte
      }
    }
  }
  //xthal_set_intenable(intLevels);
  taskENABLE_INTERRUPTS();
  //  xTaskResumeAll();
  return bitCount;
}
#endif

int64_t readTrack(boolean silent, boolean compOnly)
{
  return readTrack2(silent, compOnly, false);
}
/*
   reads a track from the floppy disk
   optional parameter silent to supress all debug messages
*/
int64_t readTrack2(boolean silent, boolean compOnly, boolean ignoreIndex)
{
  setMode(densMode);
  int j = 0;
  for (j = 0; j < retries; j++) {
    for (int i = 0; i < 256; i++) {
      hist[i] = 0;
    }
    motorOn();
    initRead();
    uint32_t tZeit = millis();
#ifdef __teensy
    watchdogReset();
    startRecord();
    while (recordOn) {
      if ((millis() - tZeit) > 300) {
        if (!silent) {
          //  Serial << "Timeout reached\n";
        }
        stopRecord();
        errors = -1;
        extError = TIMEOUT_ERR;
      }
    }
#endif
#ifdef __esp
    startRecord();
    if (espRead() == -1) {
      errors = -1;
      extError = TIMEOUT_ERR;
    }
    stopRecord();
    recordOn = false;
#endif
    tZeit = millis() - tZeit;
    if (!silent) {
      Serial.printf("ReadTime: %d\n", tZeit);
    }
    tZeit = micros();
    errors = decodeTrack(silent, compOnly, ignoreIndex);
    tZeit = micros() - tZeit;
    if (!silent) {
      Serial.printf("Decode took %d Error: %d\n", tZeit, errors);
    }
    if (getTrackInfo() != logTrack) {
      errors = -1;
      extError = SEEK_ERR;
    }
    if ((errors == 0) && (sectorCnt == sectors)) {
      extError = NO_ERR;
      break;
    }
    if (!silent) {
      Serial.printf("Read Error, retries left: %d CurrentTrack: %d error:", retries - j, logTrack);
      Serial.print(errorD | errorH, BIN);
      Serial.printf(" %d errors: %d sectorCount: %d\n", extError, errors, sectorCnt);
    }
    adjustTimings();
    // the following code tries to move the stepper / seek 0 before retrying to read the track
    // but i found out that about 6 retries are sufficient to determine if a track is bad
    // but you may enable it and set retries to 25 if you want to give it a shot
    if (!silent) {
      Serial.printf(" high2: %d high3: %d\n", high2, high3);
    }
    int tempTrack = logTrack;
    switch (j) {
      case 10:
        gotoLogicTrack(tempTrack - 2);
        gotoLogicTrack(tempTrack);
        break;
      case 15:
        gotoLogicTrack(tempTrack + 2);
        gotoLogicTrack(tempTrack);
        vTaskDelay(10);
        break;
      case 20:
        currentTrack = -1;
        gotoLogicTrack(tempTrack);
        vTaskDelay(10);
        break;
      case 25:
        currentTrack = -1;
        gotoLogicTrack(tempTrack);
        vTaskDelay(10);
        break;
      default:
        // do nothing
        break;
    }

  }
  weakTracks[logTrack] = j;
  trackLog[logTrack] = getTrackInfo();
  if (sectorCnt != sectors) {
    errors = -1;
    extError = NUMSECTOR_ERR;
  }
  return errors;
}

/*
   analyses a track from the floppy disk
   optional parameter silent to supress all debug messages
*/
int64_t analyseTrack(boolean silent, int dens)
{
  errors = 0;
  setMode(dens);
  for (int i = 0; i < 256; i++) {
    hist[i] = 0;
  }
  int j = 0;
  for (j = 0; j < 10; j++) {
    motorOn();
    initRead();
    uint32_t tZeit = millis();
#ifdef __teensy
    watchdogReset();
    startRecord();
    while (recordOn) {
      if ((millis() - tZeit) > 300) {
        if (!silent) {
          //  Serial << "Timeout reached\n";
        }
        stopRecord();
        errors = -1;
        extError = TIMEOUT_ERR;
      }
    }
#endif
#ifdef __esp
    startRecord();
    if (espRead() == -1) {
      errors = -1;
      extError = TIMEOUT_ERR;
    }
    stopRecord();
    recordOn = false;
#endif
  }
  setMode(densMode);
  return errors;
}

/*
   read Diskname from Track 80
*/
String getName()
{
  if (!noDisk) {
    String volumeName;
    gotoLogicTrack(80);
    readTrack(true, false);
    struct Sector *aSec = (Sector *) &track[0].sector;
    volumeName = "NDOS";
    int nameLen = aSec->data[432];
    if (nameLen > 30) return "NDOS";
    int temp = 0;
    for (int i = 0x04; i < 0x0c; i++) {
      temp += aSec->data[i];
    }
    for (int i = 0x10; i < 0x14; i++) {
      temp += aSec->data[i];
    }
    for (int i = 463; i < 472; i++) {
      temp += aSec->data[i];
    }
    for (int i = 496; i < 504; i++) {
      temp += aSec->data[i];
    }
    if (temp != 0) return "NDOS";
    for (int i = 0; i < 4; i++) {
      temp += aSec->data[i];
    }
    if (temp != 2) return "NDOS";
    temp = 0;
    for (int i = 508; i < 512; i++) {
      temp += aSec->data[i];
    }
    if (temp != 1) return "NDOS";
    volumeName = "";
    for (int i = 0; i < nameLen; i++) {
      volumeName += (char) aSec->data[433 + i];
    }
    return volumeName;
  } else {
    return "NODISK";
  }
}

/*
   Initializes Variables for reading a track
*/
void initRead ()
{
  bCnt = 0;
  readPtr = 0;
  bitCount = 0;
  sectorCnt = 0;
  errors = 0;
  errorH = 0;
  errorD = 0;
  extError = NO_ERR;
  for (uint32_t i = 0; i < streamLen; i++) {
    stream[i] = 0x00;
  }
#ifdef __teensy
  setupFTM0();
#endif
}

/*
   starts recording
*/
void startRecord()
{
  recordOn = true;
#ifdef __teensy
  FTM0_CNT = 0x0000; // Reset the count to zero
  FTM0_SC = timerMode;
#endif
#ifdef __esp
  timerStart(myTimer);
  TIM_LOADLO = 0;
  TIM_LOADHI = 0;
  TIM_LOADREG = 1;
#endif
}

/*
   stops recording
*/
void stopRecord()
{
  recordOn = false;
#ifdef __teensy
  FTM0_SC = 0x00; // Timer off
#endif
#ifdef __esp
  timerStop(myTimer);
#endif
}

/*
   starts the motor if it isnt running yet
*/
void motorOn()
{
  motorTime = millis();
  driveSelect();
  if (motor == false) {
    digitalWriteFast(_motor, LOW);
    motor = true;
    vTaskDelay(settings.motorSpinupDelay); // more than plenty of time to spinup motor
    if (getDensity) densityDetect();
  }
}

/*
   stops motor
*/
void motorOff()
{
  driveSelect();
  motor = false;
  digitalWriteFast(_motor, HIGH);
  delayMicroseconds(settings.motorSpindownDelay);
  driveDeselect();
}

void driveSelect()
{
#ifdef debug
  //if (!motor) Serial.println("driveSelect");
  digitalWrite(13, HIGH);
#endif
  digitalWriteFast(_drivesel, LOW);
  delayMicroseconds(settings.driveSelectDelay);
}

void driveDeselect()
{
#ifdef debug
  //if (!motor) Serial.println("driveDeselect");
  digitalWrite(13, LOW);
#endif
  digitalWriteFast(_drivesel, HIGH);
  delayMicroseconds(settings.driveDeselectDelay);
}

/*
   selects travel direction of head
   dir = 0   outwards to track 0
   dir !=0 inwards to track 79
*/
void setDir(int dir)
{
  floppyPos.dir = dir;
  if (dir == 0) {
    digitalWriteFast(_dir, HIGH);
  } else {
    digitalWriteFast(_dir, LOW);
  }
  delayMicroseconds(settings.setDirDelay);
}

/*
   selects side to read/write
   side = 0  upper side
   side != 0 lower side
*/
void setSide(int side)
{
  if (side == 0) {
    digitalWriteFast(_side, HIGH);
  } else {
    digitalWriteFast(_side, LOW);
  }
  delayMicroseconds(settings.setSideDelay);
  floppyPos.side = side;
}

/*
   steps one track into the direction selected by setDir()
*/
void step1()
{
  digitalWriteFast(_step, LOW);
  delayMicroseconds(settings.stepDelay);
  digitalWriteFast(_step, HIGH);
  vTaskDelay(settings.stepSettleDelay);
  if (floppyPos.dir == 0) {
    floppyPos.track--;
  } else {
    floppyPos.track++;
  }
}

/*
   move head to track 0
*/
int seek0()
{
  int trkCnt = 0;
  setDir(0);
  while (digitalRead(_track0) == 1) {
    step1();
    trkCnt++;
    if (trkCnt > 85) {
      extError = TRACK0_ERR;
      return -1;
    }
  }
  currentTrack = 0;
  floppyPos.track = 0;
  extError = NO_ERR;
  return 0;
}

/*
   moves head to physical track (0-xx)
*/
int gotoTrack(int track)
{
  motorOn();
  if (track == 0) {
    currentTrack = -1;
  }
  int steps = 0;
  if (track < 0) return -1;
  if (currentTrack == -1) {
    if (seek0() == -1) {
      return -1;
    }
  }
  if (track == currentTrack) {
    return 0;
  }
  if (track < currentTrack) {
    setDir(0);
    steps = currentTrack - track;
    currentTrack = track;
  } else {
    setDir(1);
    steps = track - currentTrack;
    currentTrack = track;
  }
  for (int i = 0; i < steps; i++) {
    step1();
  }
  return 0;
}

/*
   moves head to logical amiga track (0-159)
*/
void gotoLogicTrack(int track)
{
  if (track > 164) track = 164;
  logTrack = track;
  setSide(track % 2);
  gotoTrack(track / 2);
  vTaskDelay(settings.gotoTrackSettle);
}

/*
   prints some stuff, mostly for debugging disk signals
*/
void printStatus()
{
  driveSelect();
  setDir(0);
  step1();
  seek0();
  Serial.printf("Trk0: %d WProt: %d Disk Change: %d Track: %d Side: %d Dir: %d\n",
                (digitalRead(_track0)), (digitalRead(_wprot)), (digitalRead(_diskChange)),
                floppyPos.track, floppyPos.side, floppyPos.dir);
  Serial.printf("Density: %d getDensity: %d WriteSize: %d StreamLen: %d Motor: %d\n",
                densMode, getDensity, writeSize, streamLen, motor);
}

boolean writeProtect()
{
  if (digitalRead(_wprot) == 0)
    wProt = true;
  else
    wProt = false;
  return wProt;
}

/*
   checks if disk has changed / is inserted, do one step and check status again because the signal gets
   updated after one step
   returns 1 = disk is in drive, 0 = do disk in drive
*/
int diskChange() // returns if a disk is inserted: 0 = no disk, 1 = disk inserted
{
  int return_value = 0;
  driveSelect();
  if (digitalRead(_diskChange) == 1) {
    if (writeProtect())
      wProt = true;
    else
      wProt = false;
    return_value = 1;
  } else {
    seek0();
    setDir(0);
    step1();
    if (writeProtect())
      wProt = true;
    else
      wProt = false;
    return_value = digitalRead(_diskChange);
  }
  if (return_value)
    noDisk = false;
  else
    noDisk = true;
  if (noDisk) driveDeselect();
  return return_value;
}

void diskChangeIRQ()
{
  if (!dchgActive) return;
  //  digitalWrite(13, HIGH);
  if (motor == true) motorOff();
#ifdef hdmode
  getDensity = true;
#endif
  noDisk = true;
  if (mtpOn) {
    if (ADFisOpen) {
      mtpd.sendEvent(MTP_EVENT_STORE_REMOVED, FDid);
      mtpd.sendEvent(MTP_EVENT_STORE_REMOVED, FSid);
      closeDrive();
    }
  }
  dchgActive = false;
}

void bitCounter()
{
  bitCount++;
}

/*
   measures time for one rotation of the disk, returns milliseconds and
   updates the bitCounter for detecting HD formatted disks
*/
int indexTimer()
{
  motorOn();
  attachInterrupt(_readdata, bitCounter, FALLING);
  uint32_t tRead = micros();
  bitCount = 0;
  vTaskDelay(200);
  detachInterrupt(_readdata);
  tRead = micros() - tRead;
  return (tRead / 1000);
}

int findMinima(int start)
{
  int first = 0;
  int last = 0;
  int tMin = 500;
  for (int i = -30; i < 30; i++) {
    if (hist[i + start] < tMin) {
      tMin = hist[i + start];
      first = i + start;
    }
    if (hist[i + start] == tMin) {
      tMin = hist[i + start];
      last = i + start;
    }
  }
  return (first + last) / 2;
}

/*
   adjusts the timings for the transitions lengths, that way there is a better chance
   to read disks written by drives with too slow / fast rpm
*/
void adjustTimings()
{
  setTimings(densMode);
  high2 = findMinima(high2);
  high3 = findMinima(high3);
}

/*
   prints histogram of last read track in ascii
   mainly for debugging purposes
*/
void printHist(int scale)
{
  float zeit;
  uint32_t summe = 0;
  uint16_t maxScale = 0;
  for (int i = 0; i < 256; i++)
    if (hist[i] > maxScale) maxScale = hist[i];
  if (scale == -1) scale = maxScale / 70;
  for (int i = 0; i < 256; i++) {
    if (hist[i] > 0) {
      summe = summe + (hist[i] * i);
#ifdef __esp
      zeit = (float(i) * myTimerDiv / 80 );
#else
      if (densMode == HD)
        zeit = (float(i) * 0.02083333) + 0.0625 + 0.2292;
      else
        zeit = (float(i) * 0.04166667) + 0.0625 + 0.2292;
      // (count * time per count) + FTMx Delay (3 SysClk) + ISR Overhead to restart timer
#endif
      Serial.print(zeit);
      Serial.printf(":%3d-%5d|", i, hist[i]);
      for (int j = 0; j < (hist[i] / scale); j++) {
        Serial.print("+");
      }
      Serial.println();
    }
  }
  Serial.printf("Summe: %ld\n", summe);
  Serial.printf("1. Minima: %d high2: %d\n", findMinima(high2), high2);
  Serial.printf("2. Minima: %d high3: %d\n", findMinima(high3), high3);
}

/*
   outputs the histogram of flux transistions in binary form
*/
void printFlux()
{
  byte a, b, c, d;
  for (int i = 0; i < 256; i++) {
    a = hist[i];
    b = hist[i] >> 8;
    c = hist[i] >> 16;
    d = hist[i] >> 24;
    Serial.write(a);
    Serial.write(b);
    Serial.write(c);
    Serial.write(d);
  }
}

/*
   counts the transistions and calculates the real read bits of the last read track
   mainly for debugging
*/
void analyseHist(boolean silent)
{
  uint32_t trackLen = 0;
  uint32_t transitions = 0;
  int samp = 0;
  for (int i = 0; i < 256; i++) {
    samp = hist[i];
    if ((i >= low2) && (i <= high2)) {
      trackLen += 2 * samp;
      transitions += samp;
    }
    if ((i >= high2 + 1) && (i <= high3)) {
      trackLen += 3 * samp;
      transitions += samp;
    }
    if ((i >= high3 + 1) && (i <= high4)) {
      trackLen += 4 * samp;
      transitions += samp;
    }
  }
  if (silent == false) {
    Serial.print("Transitions: ");
    Serial.print(transitions);
    Serial.print(" Real Bits: ");
    Serial.println(trackLen);
  }
}

/*
   decodes one MFM encoded Sector into Amiga Sector
   partly based on DecodeSectorData and DecodeLongword from AFR.C, written by
   Marco Veneri Copyright (C) 1997 released as public domain
*/
int decodeSector(uint32_t secPtr, int index, boolean silent, boolean ignoreIndex)
{
  secPtr += 8; // skip sync and magic word
  uint32_t tmp[4];
  uint32_t decoded;
  uint32_t chkHeader = 0;
  uint32_t chkData = 0;
  //decode format, track, sector, distance 2 gap
  for (int i = 0; i < 1; i++) {
    tmp[0] = ((stream[secPtr + (i * 8) + 0] << 8) + stream[secPtr + (i * 8) + 1]) & 0x5555;
    tmp[1] = ((stream[secPtr + (i * 8) + 2] << 8) + stream[secPtr + (i * 8) + 3]) & 0x5555;
    tmp[2] = ((stream[secPtr + (i * 8) + 4] << 8) + stream[secPtr + (i * 8) + 5]) & 0x5555;
    tmp[3] = ((stream[secPtr + (i * 8) + 6] << 8) + stream[secPtr + (i * 8) + 7]) & 0x5555;

    // even bits
    tmp[0] = (tmp[0] << 1);
    tmp[1] = (tmp[1] << 1);

    // or with odd bits
    tmp[0] |= tmp[2];
    tmp[1] |= tmp[3];

    // final longword
    decoded = ((tmp[0] << 16) | tmp[1]);

    sectorTable[index].sector = (decoded >> 8) & 0xff;
    if (!ignoreIndex) {
      index = (decoded >> 8) & 0xff;
      // if sector out of bounds, return with error
      if ((index + 1 > sectors) || (index < 0)) {
        errors = errors | (1 << 31);
        extError = SECBOUNDS_ERR;
        return -1;
      }
    }
    tempSector[(i * 4) + 0] = decoded >> 24; // format type 0xff = amiga
    tempSector[(i * 4) + 1] = decoded >> 16; // track
    tempSector[(i * 4) + 2] = decoded >> 8;  // sector
    tempSector[(i * 4) + 3] = decoded;       // distance to gap
  }
  //decode checksums
  for (int i = 5; i < 7; i++) {
    tmp[0] = ((stream[secPtr + (i * 8) + 0] << 8) + stream[secPtr + (i * 8) + 1]) & 0x5555;
    tmp[1] = ((stream[secPtr + (i * 8) + 2] << 8) + stream[secPtr + (i * 8) + 3]) & 0x5555;
    tmp[2] = ((stream[secPtr + (i * 8) + 4] << 8) + stream[secPtr + (i * 8) + 5]) & 0x5555;
    tmp[3] = ((stream[secPtr + (i * 8) + 6] << 8) + stream[secPtr + (i * 8) + 7]) & 0x5555;
    // even bits
    tmp[0] = (tmp[0] << 1);
    tmp[1] = (tmp[1] << 1);
    // or with odd bits
    tmp[0] |= tmp[2];
    tmp[1] |= tmp[3];
    // final longword
    decoded = ((tmp[0] << 16) | tmp[1]);
    tempSector[(i * 4) + 0] = decoded >> 24;
    tempSector[(i * 4) + 1] = decoded >> 16;
    tempSector[(i * 4) + 2] = decoded >> 8;
    tempSector[(i * 4) + 3] = decoded;
    // store checksums for later use
    if (i == 5) {
      chkHeader = decoded;
    } else {
      chkData = decoded;
    }
  }
  // decode all the even data bits
  uint32_t data;
  for (int i = 0; i < 256; i++)
  {
    data = ((stream[secPtr + (i * 2) + 56] << 8) + stream[secPtr + (i * 2) + 57]) & 0x5555;
    tempSector[(i * 2) + 28] = (unsigned char)(data >> 7);
    tempSector[(i * 2) + 29] = (unsigned char)(data << 1);
  }

  // or with odd data bits
  for (int i = 0; i < 256; i++)
  {
    data = ((stream[secPtr + (i * 2) + 56 + 512] << 8) + stream[secPtr + (i * 2) + 57 + 512]) & 0x5555;
    tempSector[(i * 2) + 28] |= (unsigned char)(data >> 8);
    tempSector[(i * 2) + 29] |= (unsigned char)(data);
  }
  // check für checksum errors and generate error flags
  if (calcChkSum(secPtr, 0, 40) != chkHeader) {
    errorH = errorH | (1 << index);
    if (!silent) Serial.print("H");
    extError = HEADERCHKSUM_ERR;
  }
  if (calcChkSum(secPtr, 56, 1024) != chkData) {
    errorD = errorD | (1 << index);
    if (!silent) Serial.print("D");
    extError = DATACHKSUM_ERR;
  }
  return index;
}

/*
   decodes one MFM encoded Sector into Dos Sector
*/
int decodeDos(uint32_t secPtr, int index, boolean silent, boolean ignoreIndex)
{
  secPtr += 8; // skip sync and magic word

  uint32_t data;
  uint8_t outByte;
  for (int i = 0; i < 512; i++)
  {
    outByte = 0;
    data = ((stream[secPtr + (i * 2)] << 8) + stream[secPtr + (i * 2) + 1]);
    for (int j = 0; j < 8; j++)
    {
      if ((data & 0x8000) != 0) outByte = outByte | 0x80;
      outByte = outByte >> 1;
      data = data << 2;
    }
    tempSector[i + 28] = outByte;
  }
  return index;
}

/*
   calculates a checksum of <secPtr> at <pos> for <b> bytes length
   returns checksum
*/
uint32_t calcChkSum(uint32_t secPtr, int pos, int b)
{
  uint32_t chkSum = 0;
  uint32_t tSum = 0;
  for (int i = 0; i < b / 4; i++) {
    tSum = stream[secPtr + (i * 4) + pos + 0];
    tSum = tSum << 8;
    tSum += stream[secPtr + (i * 4) + pos + 1];
    tSum = tSum << 8;
    tSum += stream[secPtr + (i * 4) + pos + 2];
    tSum = tSum << 8;
    tSum += stream[secPtr + (i * 4) + pos + 3];
    chkSum = chkSum ^ tSum;
  }
  chkSum = chkSum & 0x55555555;
  return chkSum;
}

/*
   decodes a whole track
   optional parameter silent to suppress all serial debug info
*/
int decodeTrack(boolean silent, boolean compOnly, boolean ignoreIndex) {
  if (!silent) {
    Serial.printf("Decoding Bitstream, expecting %d Sectors.\n", sectorCnt);
    Serial.print("Sectors start at: ");
  }
  int index = 0;
  for (uint32_t i = 0; i < sectorCnt; i++) {
    if (!silent) {
      Serial.print(sectorTable[i].bytePos);
    }
    if (i != sectorCnt - 1) {
      if (!silent) {
        Serial.print(", ");
      }
    }
    if (ignoreIndex)
      index = decodeDos(sectorTable[i].bytePos, i, silent, ignoreIndex);
    else
      index = decodeSector(sectorTable[i].bytePos, i, silent, ignoreIndex);
    if (index == -1) return errorD | errorH;
    if (compOnly) {
      if (memcmp(track[index].sector, tempSector, 540) != 0) return -1;
    }
    else {
      memcpy(track[index].sector, tempSector, 540);
    }
  }
  if (!silent) {
    Serial.println();
  }
  errors = errorD | errorH;
  return errors;
}

/*
   returns current track number from decoded sectors in buffer
*/
int getTrackInfo()
{
  int tTrack = 0;
  for (uint32_t i = 0; i < sectorCnt; i++) {
    tTrack = tTrack + track[i].sector[1];
  }
  return tTrack / sectors;
}

/*
   dumps the sector <index> from the buffer in human readable acsii to the serial port
   mainly for debugging
*/
void printAmigaSector(int index)
{
  struct Sector *aSec = (Sector *) &track[index].sector;
  Serial.printf("Format Type: %d Track: %d Sector: %d NumSec2Gap: %d Data Chk: ",
                aSec->format_type, aSec->track, aSec->sector, aSec->toGap);
  Serial.print(aSec->data_chksum, HEX);
  Serial.printf(" Header Chk: ");
  Serial.println(aSec->header_chksum, HEX);
  for (int i = 0; i < 16; i++) {
    for (int j = 0; j < 32; j++) {
      if (aSec->data[(i * 32) + j] < 16) {
        Serial.print("0");
      }
      Serial.print(aSec->data[(i * 32) + j], HEX);
      Serial.print(" ");
    }
    for (int j = 0; j < 32; j++) {
      Serial.print(byte2char(aSec->data[(i * 32) + j]));
    }
    Serial.println();
  }
}

/*
   dumps the whole track in ascii
   mainly for debugging
*/
void printTrack()
{
  for (uint32_t i = 0; i < sectorCnt; i++) {
    printAmigaSector(i);
  }
}

/*
   dumps the data section of a sector in binary format
*/
void dumpSector(int index)
{
  struct Sector *aSec = (Sector *) &track[index].sector;
  /*  for (int i = 0; i < 512; i++) {
      Serial.print((char)aSec->data[i]);
    }*/
  Serial.write(aSec->data, 512);
}

/*
   returns a pointer to the sector
*/
byte* ptrSector(int index)
{
  struct Sector *aSec = (Sector *) &track[index].sector;
  return &aSec->data[0];
}

/*
   reads a sector from the serial in binary format
*/
int loadSector(int index)
{
  char tBuffer[512];
  struct Sector *aSec = (Sector *) &track[index].sector;
  int rByte = Serial.readBytes(tBuffer, 512);
  if (rByte != 512) return -1;
  for (int i = 0; i < 512; i++) {
    aSec->data[i] = tBuffer[i];
  }
  return 0;
}

/*
   dumps the whole track in binary form to the serial port
*/
void downloadTrack()
{
  for (int i = 0; i < sectors; i++) {
    dumpSector(i);
  }
}

/*
   reads a whole track from the serial port in binary form
*/
void uploadTrack()
{
  errors = 0;
  extError = NO_ERR;
  for (int i = 0; i < sectors; i++) {
    if (loadSector(i) != 0) {
      extError = UPLOAD_ERR;
      errors = -1;
      return;
    }
  }
}

/*
   returns c if printable, else returns a whitespace
*/
char byte2char(byte c) {
  if ((c < 32) | (c > 126)) {
    return (char) 46;
  } else {
    return (char) c;
  }
}

/*
   prints v in ascii encoded binary
*/
void print_binary(int v, int num_places)
{
  int mask = 0, n;
  for (n = 1; n <= num_places; n++)
  {
    mask = (mask << 1) | 0x0001;
  }
  v = v & mask;  // truncate v
  while (num_places)
  {
    if (v & (0x0001 << (num_places - 1)))
    {
      Serial.print("1");
    }
    else
    {
      Serial.print("0");
    }
    --num_places;
  }
}

/*
   prints v in ascii encoded binary reversed
   lowest bit first
*/
void print_binary_rev(uint32_t v, uint32_t num_places)
{
  for (uint32_t i = 0; i < num_places; i++)
  {
    if (v & 0x01)
      Serial.print("1");
    else
      Serial.print("0");
    v = v >> 1;
  }
}

/*
   fills a sector with "ADFCopy"
*/
void fillSector(int sect, boolean fillType)
{
  char adfTag[8] = "ADFCopy";
  for (int i = 0; i < 64; i++) {
    if (fillType) {
      memcpy(&track[sect].sector[i * 8 + 28], &adfTag, 8);
    } else {
      memset(&track[sect].sector[i * 8 + 28], 0, 8);
    }
  }
}

/*
   encodes odd bits of a longword to mfm
*/
uint32_t oddLong(uint32_t odd)
{
  odd = ((odd >> 1) & MFM_MASK);
  odd = odd | ((((odd ^ MFM_MASK) >> 1) | 0x80000000) & ((odd ^ MFM_MASK) << 1));
  return odd;
}

/*
   encodes even bits of a longword to mfm
*/
uint32_t evenLonger(uint32_t even)
{
  even = (even & MFM_MASK);
  even = even | ((((even ^ MFM_MASK) >> 1) | 0x80000000) & ((even ^ MFM_MASK) << 1));
  return even;
}

/*
   encodes one byte <curr> into mfm, takes into account if previous byte ended with a set bit
   returns the two mfm bytes as a word
*/
uint16_t mfmByte(byte curr, uint16_t prev)
{
  byte even = ((curr >> 1) & 0x55);
  even = even | ((((even ^ MFM_MASK) >> 1) | 0x8000000) & (((even ^ MFM_MASK) << 1)));
  if ((prev & 0x0001) == 1) {
    even = even & 0x7f;
  }
  byte odd = (curr & 0x55);
  odd = odd | ((((odd ^ MFM_MASK) >> 1) | 0x8000000) & (((odd ^ MFM_MASK) << 1)));
  if ((prev & 0x0100) == 0x0100) {
    odd = odd & 0x7f;
  }
  return (odd << 8) | even;
}

/*
   stores a longword in the byte array
*/
void putLong(uint32_t tLong, byte * dest)
{
  *(dest + 0) = (unsigned byte) ((tLong & 0xff000000) >> 24);
  *(dest + 1) = (unsigned byte) ((tLong & 0xff0000) >> 16);
  *(dest + 2) = (unsigned byte) ((tLong & 0xff00) >> 8);
  *(dest + 3) = (unsigned byte) ((tLong & 0xff));
}

/*
   encodes a sector into a mfm bitstream
*/
void encodeSector(uint32_t tra, uint32_t sec, byte * src, byte * dest)
{
  uint32_t tmp, headerChkSum, dataChkSum;
  uint16_t curr;
  uint16_t prev;
  byte prevByte;
  struct Sector *aSec = (Sector *) (src - 28);

  // write sync and magic word
  putLong(0xaaaaaaaa, dest + 0);
  putLong(0x44894489, dest + 4);

  // format, track, sector, distance to gap
  tmp = 0xff000000 | (tra << 16) | (sec << 8) | (sectors - sec);
  putLong(oddLong(tmp), dest + 8);
  putLong(evenLonger(tmp), dest + 12);
  // update the source sector
  aSec->format_type = 0xff;
  aSec->track = tra;
  aSec->sector = sec;
  aSec->toGap = sectors - sec;

  // fill unused space in sector header
  prevByte = *(dest + 15);
  for (int i = 16 ; i < 48; i++)
  {
    if ((prevByte & 0x01) == 1) {
      *(dest + i) = 0xaa & 0x7f;
    } else {
      *(dest + i) = 0xaa;
    }
    prevByte = *(dest + i);
  }
  //update source sector
  for (int i = 0; i < 16; i++)
    aSec->os_recovery[i] = 0;

  // data block encode
  prev = (uint16_t)prevByte;
  for (int i = 64 ; i < 576; i++)
  {
    curr = mfmByte(*(src + i - 64), prev);
    prev = curr;
    *(dest + i) = (byte) (curr & 0xff);
    *(dest + i + 512) = (byte) (curr >> 8);
  }

  // calc headerchecksum
  headerChkSum = calcChkSum(sectorTable[sec].bytePos, 8, 40);
  putLong(oddLong(headerChkSum), dest + 48);
  putLong(evenLonger(headerChkSum), dest + 52);
  // update source sector headerChkSum
  aSec->header_chksum = __bswap32(headerChkSum);

  // calc datachecksum
  dataChkSum = calcChkSum(sectorTable[sec].bytePos, 64, 1024);
  putLong(oddLong(dataChkSum), dest + 56);
  putLong(evenLonger(dataChkSum), dest + 60);
  // update source sector headerChkSum
  aSec->data_chksum = __bswap32(dataChkSum);

}

/*
   fills bitstream with 0xAA
*/
void fillTrackGap(byte * dst, int len)
{
  for (int i = 0; i < len; i++)
  {
    *dst++ = 0xaa;
  }
}

/*
   encodes a complete track + gap into mfm bitstream
   the gap gets encoded before the sectors to make sure when the track gap is too long for
   the track the first sector doesnt gets overwritten, this way only the track gap gets overwritten
   and the track contains no old sector headers
*/
void floppyTrackMfmEncode(uint32_t track_t, byte * src, byte * dst)
{
  fillTrackGap(dst, FLOPPY_GAP_BYTES);
  for (int i = 0; i < sectors; i++)
  {
    sectorTable[i].bytePos = (i * 1088) + FLOPPY_GAP_BYTES;
    encodeSector(track_t, i, src + 28 + (i * 540), dst + (i * 1088) + FLOPPY_GAP_BYTES);
  }
}


/*
   tries to detect how the teensy is connected to the drive by
   stepping the head to track0, if track0 switches to low we have a hit.
   there are two different schematics at the moment: breadboard and pcb
*/
#define pcbVer 1
#define breadboardVer 0
int hardwareVersion()
{
#ifdef __teensy
  watchdogReset();
#endif
  // check for pcb v3 & v4 pinout version, v1&v2 were prototypes
  pinMode(16, OUTPUT);           //drivesel
  digitalWriteFast(16, LOW);     //select drive
  pinMode(18, OUTPUT);           //direction
  digitalWriteFast(18, HIGH);    //-> track 0
  pinMode(19, OUTPUT);           // set step pin to output
  pinMode(7, INPUT_PULLUP);      // track 0 as input
  for (int i = 0; i < 85; i++) { // move head to track 0
    digitalWriteFast(19, LOW);
    delayMicroseconds(settings.stepDelay);
    digitalWriteFast(19, HIGH);
    vTaskDelay(settings.stepSettleDelay);
    if (digitalRead(7) == 0) return pcbVer; // track 0? -> hardware rev pcb
  }
  pinMode(16, INPUT); // switch previous used pins to input
  pinMode(18, INPUT);
  pinMode(19, INPUT);
  pinMode(7, INPUT);

  // check for breadboard pinout version
  pinMode(4, OUTPUT);            //drivesel
  digitalWrite(4, LOW);          //select drive
  pinMode(6, OUTPUT);            //direction
  digitalWriteFast(6, HIGH);     //-> track 0
  pinMode(7, OUTPUT);            // set step pin to output
  pinMode(10, INPUT_PULLUP);     // track 0 as input
  for (int i = 0; i < 85; i++) { // move head to track 0
    digitalWriteFast(7, LOW);
    delayMicroseconds(settings.stepDelay);
    digitalWriteFast(7, HIGH);
    vTaskDelay(settings.stepSettleDelay);
    if (digitalRead(10) == 0) return breadboardVer; // track 0? -> hardware rev breadboard
  }
  pinMode(4, INPUT); // switch previous used pins to input
  pinMode(6, INPUT);
  pinMode(7, INPUT);
  pinMode(10, INPUT);
  return -1; // unknown pinout or drive broken
}

/*
   sets up the registers according to the hardware version
*/
void registerSetup(int version)
{
#ifdef __teensy
  if (version == pcbVer)
  {
    //    _dens = 14;      //2 density select IN
    _index = 15;     //8 index OUT
    _drivesel = 16;  //12 drive select 1 IN
    _motor = 17;     //16 motor1 on IN
    _dir = 18;       //18 direction IN
    _step = 19;      //20 step IN
    _writedata = 9; //22 write data IN
    _writeen = 8;   //24 write enable IN
    _track0 = 7;   //26 track 0 OUT
    _wprot = 6;    //28 write protect OUT
    _readdata = 5; //30 read data OUT (FTM0_CH1) *** do not change this pin ***
    _side = 4;    //32 head select IN
    _diskChange = 3;    //34 disk change OUT

    // FlexTimerModule defines for Pin 5
    FTChannelValue = 0x40038048;
    FTStatusControlRegister = 0x40038044;
    FTPinMuxPort = 0x4004C01C;
  }
  if (version == breadboardVer)
  {
    //    _dens = 2;      //2 density select IN
    _index = 3;     //8 index OUT
    _drivesel = 4;  //12 drive select 1 IN
    _motor = 5;     //16 motor1 on IN
    _dir = 6;       //18 direction IN
    _step = 7;      //20 step IN
    _writedata = 8; //22 write data IN
    _writeen = 9;   //24 write enable IN
    _track0 = 10;   //26 track 0 OUT
    _wprot = 11;    //28 write protect OUT
    _readdata = 22; //30 read data OUT (FTM0_CH1) *** do not change this pin ***
    _side = 14;    //32 head select IN
    _diskChange = 15;    //34 disk change OUT

    // FlexTimerModule defines for Pin 22
    FTChannelValue = 0x40038010;
    FTStatusControlRegister = 0x4003800C;
    FTPinMuxPort = 0x4004B004;

  }
#endif
#ifdef __esp
  if (version == 2)
  {
    _index = GPIO_NUM_32;     //8 index OUT
    _drivesel = GPIO_NUM_33;  //12 drive select 1 IN
    _motor = GPIO_NUM_25;     //16 motor1 on IN
    _dir = GPIO_NUM_26;       //18 direction IN
    _step = GPIO_NUM_16;      //20 step IN
    _writedata = GPIO_NUM_17; //22 write data IN
    _writeen = GPIO_NUM_15;   //24 write enable IN
    _track0 = GPIO_NUM_35;   //26 track 0 OUT
    // GPIO_35 needs external Pullup 10k
    _wprot = GPIO_NUM_34;    //28 write protect OUT
    // GPIO_34 needs external Pullup 10k
    _readdata = GPIO_NUM_22; //30 read data OUT
    _side = GPIO_NUM_27;    //32 head select IN
    _diskChange = GPIO_NUM_21;    //34 disk change OUT
    pinMode(_readdata, INPUT_PULLUP); //esp

    /*
       SD Card @ ESP
       Pin 7    IO19
       Pin 6    GND
       Pin 5    IO18
       Pin 4    3,3V
       Pin 3    GND
       Pin 2    IO23
       Pin 1    IO5
    */
  }
#endif
}

/*
   Sets standard timings for Teensy 3.2
*/
#ifdef __teensy
void setTimings(int density)
{
  if (density == 1) {
    low2 = tHDlow2;
    high2 = tHDhigh2;
    high3 = tHDhigh3;
    high4 = tHDhigh4;
  } else {
    low2 = tDDlow2;
    high2 = tDDhigh2;
    high3 = tDDhigh3;
    high4 = tDDhigh4;
  }
}
#endif

/*
   Sets standard timings for ESP32
*/
#ifdef __esp
void setTimings(int density)
{
  if (density == 1) {
    low2 = 25;
    high2 = 44;
    high3 = 60;
    high4 = 255;
  } else {
    low2 = 50;
    high2 = 95;
    high3 = 135;
    high4 = 255;
  }
}
#endif

/*
   initializes variables and timings according to the density mode
*/
void setMode(int density)
{
  setTimings(density);
  if (density == 1) {
#ifdef __esp
    espWriteTimer = 80;
#endif
    sectors = 22;
    timerMode = timerModeHD;
    streamLen = streamSizeHD;
    filterSetting = filterSettingHD;
    writeSize = writeSizeHD;
    transitionTime = transTimeHD;
    if (densMode != HD) {
      if (retries < settings.hdRetries) retries = settings.hdRetries;
    }
    densMode = HD;
  } else {
#ifdef __esp
    espWriteTimer = 160;
#endif
    sectors = 11;
    timerMode = timerModeDD;
    streamLen = streamSizeDD;
    filterSetting = filterSettingDD;
    writeSize = writeSizeDD;
    transitionTime = transTimeDD;
    if (densMode != DD) {
      if (retries < settings.sdRetries) retries = settings.sdRetries;
    }
    densMode = DD;
  }
}

/*
   tries to detect the density of the disk by counting the transitions on a track
*/
void densityDetect()
{
  indexTimer();
#ifdef hdmode
  if (bitCount > 60000) {
    setMode(HD);
  } else {
    setMode(DD);
  }
#endif
  getDensity = false;
}

/*
   read function with verbose output
*/
void doRead(int track, boolean ignoreIndex)
{
  gotoLogicTrack(track);
  errors = readTrack2(false, false, ignoreIndex);
  if (errors != -1) {
    Serial.printf("Sectors found: %d Errors found: ", sectorCnt);
    Serial.print(errorD | errorH, BIN);
    Serial.println();
    Serial.printf("Track expected: %d Track found: %d\n", param.toInt(), getTrackInfo());
    Serial.printf("high2 %d high3 %d bitCount: %d\n", high2, high3, bitCount);
    Serial.println("OK");
  } else {
    Serial.printf("Sectors found: %d Errors found: ", sectorCnt);
    Serial.print(errorD | errorH, BIN);
    Serial.println();
    Serial.printf("Track expected: %d Track found: %d\n", param.toInt(), getTrackInfo());
    Serial.printf("high2 %d high3 %d bitCount: %d\n", high2, high3, bitCount);
    Serial.println("Read failed!");
  }
}

/*
   returns the creation and modified date of the disk from rootblock
*/
void diskDates(char* cdate, char* ldate)
{
  tmElements_t timeTemp;
  if (logTrack != 80) {
    gotoLogicTrack(80);
    readTrack(true, false);
  }
  struct Sector *rootTemp = (Sector *) &track[0].sector;
  struct rootBlock *root = (rootBlock *) &rootTemp->data;

  time_t tZeit = (__bswap32(root->days) + 2922) * 24 * 60 * 60
                 + (__bswap32(root->mins) * 60)
                 + (__bswap32(root->ticks) / 50);
  breakTime(tZeit, timeTemp);
  sprintf(ldate, "%4d%02d%02dT%02d%02d%02d.0\n",
          timeTemp.Year + 1970,
          timeTemp.Month,
          timeTemp.Day,
          timeTemp.Hour,
          timeTemp.Minute,
          timeTemp.Second);

  tZeit = (__bswap32(root->c_days) + 2922) * 24 * 60 * 60
          + (__bswap32(root->c_mins) * 60)
          + (__bswap32(root->c_ticks) / 50);
  breakTime(tZeit, timeTemp);
  sprintf(cdate, "%4d%02d%02dT%02d%02d%02d.0\n",
          timeTemp.Year + 1970,
          timeTemp.Month,
          timeTemp.Day,
          timeTemp.Hour,
          timeTemp.Minute,
          timeTemp.Second);
}

/*
   prints various stuff from the root and bootblock for debugging
*/
void diskInfoOld()
{
  tmElements_t timeTemp;
  gotoLogicTrack(80);
  readTrack(true, false);
  struct Sector *rootTemp = (Sector *) &track[0].sector;
  struct rootBlock *root = (rootBlock *) &rootTemp->data;

  Serial.printf("type: %d\n", (__bswap32(root->type)));
  Serial.printf("header_key: %d\n", (__bswap32(root->header_key)));
  Serial.printf("high_seq: %d\n", (__bswap32(root->high_seq)));
  Serial.printf("ht_size: %d\n", (__bswap32(root->ht_size)));
  Serial.printf("first_data: %d\n", (__bswap32(root->first_data)));
  Serial.printf("chksum: %x\n", (__bswap32(root->chksum)));

  unsigned long newsum;
  root->chksum = 0;
  newsum = 0L;
  for (int i = 0; i < 128; i++)
    newsum += rootTemp->data[i * 4] << 24
              | rootTemp->data[i * 4 + 1] << 16
              | rootTemp->data[i * 4 + 2] << 8
              | rootTemp->data[i * 4 + 3];
  newsum = -newsum;
  Serial.printf("newsum: %x\n", newsum);

  Serial.print("ht[72]\n");
  for (int i = 0; i < 72; i++) {
    Serial.print(__bswap32(root->ht[i]));
    Serial.print(" ");
    if ((i + 1) % 16 == 0) Serial.println();
  }
  Serial.println();
  Serial.printf("bm_flag: %d\n", (long)__bswap32(root->bm_flag));
  Serial.printf("bm_pages[25]\n");
  for (int i = 0; i < 25; i++) {
    Serial.print(__bswap32(root->bm_pages[i]));
    Serial.print(" ");
    if ((i + 1) % 16 == 0) Serial.println();
  }
  Serial.println();
  Serial.printf("bm_ext: %x\n", (__bswap32(root->bm_ext[0])));
  Serial.printf("name_len[%d]: ", root->name_len);
  for (int i = 0; i < root->name_len; i++) {
    Serial.print(root->diskname[i]);
  }
  Serial.println();
  time_t tZeit = (__bswap32(root->days) + 2922) * 24 * 60 * 60
                 + (__bswap32(root->mins) * 60)
                 + (__bswap32(root->ticks) / 50);
  breakTime(tZeit, timeTemp);
  Serial.printf("Last access: %4d%02d%02dT%02d%02d%02d.0\n",
                timeTemp.Year + 1970,
                timeTemp.Month,
                timeTemp.Day,
                timeTemp.Hour,
                timeTemp.Minute,
                timeTemp.Second);
  tZeit = (__bswap32(root->c_days) + 2922) * 24 * 60 * 60
          + (__bswap32(root->c_mins) * 60)
          + (__bswap32(root->c_ticks) / 50);
  breakTime(tZeit, timeTemp);
  Serial.printf("Created: %4d%02d%02dT%02d%02d%02d.0\n",
                timeTemp.Year + 1970,
                timeTemp.Month,
                timeTemp.Day,
                timeTemp.Hour,
                timeTemp.Minute,
                timeTemp.Second);
  Serial.printf("next_hash: %x\n", (__bswap32(root->next_hash)));
  Serial.printf("parent_dir: %x\n", (__bswap32(root->parent_dir)));
  Serial.printf("extension: %d\n", (__bswap32(root->extension)));
  Serial.printf("sec_type: %x\n", (__bswap32(root->sec_type)));

  struct Sector *bitmapTemp = (Sector *) &track[1].sector;
  struct bitmapBlock *bitmap = (bitmapBlock *) &bitmapTemp->data;
  Serial.printf("bitmap checksum: %x\n", (__bswap32(bitmap->checksum)));
  newsum = 0L;
  for (int i = 1; i < 128; i++)
    newsum -= bitmapTemp->data[i * 4] << 24
              | bitmapTemp->data[i * 4 + 1] << 16
              | bitmapTemp->data[i * 4 + 2] << 8
              | bitmapTemp->data[i * 4 + 3];
  Serial.printf("bitmap newsum: %x\n", newsum);
  uint32_t bCount = 0;
  if (densMode == HD) {
    for (int i = 0; i < 110; i++) {
      bCount += __builtin_popcount(bitmap->map[i]);
    }
  } else {
    for (int i = 0; i < 55; i++) {
      bCount += __builtin_popcount(bitmap->map[i]);
    }
  }
  Serial.printf("Blocks Free: %d\n", bCount);
  gotoLogicTrack(0);
  readTrack(true, false);
  struct Sector *bootBlock = (Sector *) &track[0].sector;
  uint32_t temp = 0;
  temp = bootBlock->data[0] << 24 | bootBlock->data[1] << 16 | bootBlock->data[2] << 8 | bootBlock->data[3];
  switch (temp) {
    case 0x444F5300:
      Serial.println("OFS");
      break;
    case 0x444F5301:
      Serial.println("FFS");
      break;
    case 0x444F5302:
      Serial.println("OFS Intl.");
      break;
    case 0x444F5303:
      Serial.println("FFS Intl.");
      break;
    case 0x444F5304:
      Serial.println("OFS DirCache");
      break;
    case 0x444F5305:
      Serial.println("FFS DirCache");
      break;
    case 0x444F5306:
      Serial.println("OFS Intl DirCache");
      break;
    case 0x444F5307:
      Serial.println("FFS Intl DirCache");
      break;
    default:
      Serial.println("NDOS");
      break;
  }
  Serial.println(temp, HEX);
}

void diskInfo()
{
  uint32_t j, newsum;
  tmElements_t timeTemp;
  gotoLogicTrack(0);
  readTrack(true, false);
  if (extError != NO_ERR) {
    Serial.println("BootChecksum: Invalid");
    Serial.println("Rootblock->: 0");
    Serial.println("Filesystem: Blank?");
    Serial.println("RootBlocktype: Invalid");
    Serial.println("RootblockChecksum: Invalid");
    Serial.println("Name: Unreadable Disk");
    Serial.println("Modified: Unknown");
    Serial.println("Created: Unknown");
    Serial.println("BitmapChecksum: Invalid");
    Serial.println("Blocks Free: Unknown");
    return;
  }
  struct Sector *bootBlock = (Sector *) &track[0].sector;
  uint32_t *tLong = (uint32_t *) &track[0].sector;
  uint32_t oldBootChkSum = __bswap32(tLong[8]);
  tLong[8] = 0;
  uint32_t bootChkSum = 0;
  for (int i = 0; i < 128; i++) {
    j = uint32_t(__bswap32(tLong[i + 7]));
    if ( (0xffffffffU - bootChkSum) < j )
      bootChkSum++;
    bootChkSum += j;
  }
  for (int i = 0; i < 128; i++) {
    j = uint32_t(__bswap32(tLong[i + 142]));
    if ( (0xffffffffU - bootChkSum) < j )
      bootChkSum++;
    bootChkSum += j;
  }
  bootChkSum = ~bootChkSum;
  tLong[8] = bootChkSum;
  if (oldBootChkSum == bootChkSum)
    Serial.println("BootChecksum: Valid");
  else
    Serial.println("BootChecksum: Invalid");
  j = uint32_t(__bswap32(tLong[9]));
  Serial.printf("Rootblock->: %d\n", j);
  j = bootBlock->data[0] << 24 | bootBlock->data[1] << 16 | bootBlock->data[2] << 8 | bootBlock->data[3];
  Serial.print("Filesystem: ");
  switch (j) {
    case 0x444F5300:
      Serial.println("OFS");
      break;
    case 0x444F5301:
      Serial.println("FFS");
      break;
    case 0x444F5302:
      Serial.println("OFS Intl.");
      break;
    case 0x444F5303:
      Serial.println("FFS Intl.");
      break;
    case 0x444F5304:
      Serial.println("OFS DirCache");
      break;
    case 0x444F5305:
      Serial.println("FFS DirCache");
      break;
    case 0x444F5306:
      Serial.println("OFS Intl DirCache");
      break;
    case 0x444F5307:
      Serial.println("FFS Intl DirCache");
      break;
    default:
      Serial.println("NDOS");
      break;
  }
  gotoLogicTrack(80);
  readTrack(true, false);
  struct Sector *rootTemp = (Sector *) &track[0].sector;
  struct rootBlock *root = (rootBlock *) &rootTemp->data;

  if ((__bswap32(root->type)) != 2)
    Serial.println("RootBlocktype: Invalid");
  else
    Serial.println("RootBlocktype: Valid");

  j = __bswap32(root->chksum);
  root->chksum = 0;
  newsum = 0;
  for (int i = 0; i < 128; i++)
    newsum += rootTemp->data[i * 4] << 24
              | rootTemp->data[i * 4 + 1] << 16
              | rootTemp->data[i * 4 + 2] << 8
              | rootTemp->data[i * 4 + 3];
  newsum = -newsum;
  //  Serial.printf("%08x %08x\n", j, newsum);
  if (newsum == j)
    Serial.println("RootblockChecksum: Valid");
  else
    Serial.println("RootblockChecksum: Invalid");

  Serial.print("Name: ");
  for (int i = 0; i < root->name_len; i++) {
    Serial.print(root->diskname[i]);
  }
  Serial.println();
  time_t tZeit = (__bswap32(root->days) + 2922) * 24 * 60 * 60
                 + (__bswap32(root->mins) * 60)
                 + (__bswap32(root->ticks) / 50);
  breakTime(tZeit, timeTemp);
  Serial.printf("Modified: %02d:%02d:%02d %02d.%02d.%04d\n",
                timeTemp.Hour,
                timeTemp.Minute,
                timeTemp.Second,
                timeTemp.Day,
                timeTemp.Month,
                timeTemp.Year + 1970);
  tZeit = (__bswap32(root->c_days) + 2922) * 24 * 60 * 60
          + (__bswap32(root->c_mins) * 60)
          + (__bswap32(root->c_ticks) / 50);
  breakTime(tZeit, timeTemp);
  Serial.printf("Created: %02d:%02d:%02d %02d.%02d.%04d\n",
                timeTemp.Hour,
                timeTemp.Minute,
                timeTemp.Second,
                timeTemp.Day,
                timeTemp.Month,
                timeTemp.Year + 1970);
  struct Sector *bitmapTemp = (Sector *) &track[1].sector;
  struct bitmapBlock *bitmap = (bitmapBlock *) &bitmapTemp->data;
  newsum = 0L;
  for (int i = 1; i < 128; i++)
    newsum -= bitmapTemp->data[i * 4] << 24
              | bitmapTemp->data[i * 4 + 1] << 16
              | bitmapTemp->data[i * 4 + 2] << 8
              | bitmapTemp->data[i * 4 + 3];
  if (newsum == (__bswap32(bitmap->checksum)))
    Serial.println("BitmapChecksum: Valid");
  else
    Serial.println("BitmapChecksum: Invalid");
  uint32_t bCount = 0;
  if (densMode == HD) {
    for (int i = 0; i < 110; i++) {
      bCount += __builtin_popcount(bitmap->map[i]);
    }
  } else {
    for (int i = 0; i < 55; i++) {
      bCount += __builtin_popcount(bitmap->map[i]);
    }
  }
  Serial.printf("Blocks Free: %d\n", bCount);
}

void dumpBitmap()
{
  gotoLogicTrack(80);
  readTrack(true, false);
  struct Sector *bitmapTemp = (Sector *) &track[1].sector;
  struct bitmapBlock *bitmap = (bitmapBlock *) &bitmapTemp->data;
  Serial.print("00"); //bootblocks are always occupied and not included in bitmap
  if (densMode == HD) {
    for (int i = 0; i < 109; i++) {
      print_binary_rev(__bswap32(bitmap->map[i]), 32);
    }
    print_binary_rev(__bswap32(bitmap->map[109]), 30);
  } else {
    for (int i = 0; i < 54; i++) {
      print_binary_rev(__bswap32(bitmap->map[i]), 32);
    }
    print_binary_rev(__bswap32(bitmap->map[54]), 30);
  }
  Serial.println();
}

/*
   formats ONE track
*/
int formatTrack(int tTrack, int wRetries, boolean verify)
{
#ifdef debug
  Serial.printf("Formatting Track %d\n", tTrack);
  uint32_t zeit = millis();
#endif
  for (int i = 0; i < sectors; i++) {
    fillSector(i, true); // add ADFCopy TAG
  }
  // special handling if track = 0 -> bootblock
  if (tTrack == 0) {
    track[0].sector[28] = 0x44; //"D";
    track[0].sector[29] = 0x4f; //"O";
    track[0].sector[30] = 0x53; //"S";
    track[0].sector[31] = 0x00;
    for (int i = 32; i < 40; i++) {
      track[0].sector[i] = 0;
    }
  }
  // special handling for rootblock and bitmap
  if (tTrack == 80) {
    fillSector(0, false);
    fillSector(1, false);
    // construction of rootblock
    struct Sector *rootTemp = (Sector *) &track[0].sector;
    struct rootBlock *root = (rootBlock *) &rootTemp->data;
    root->type = __bswap32(2);
    root->ht_size = __bswap32(72);
    root->bm_flag = __bswap32(-1);
    if (densMode == HD) root->bm_pages[0] = __bswap32(1761);
    else root->bm_pages[0] = __bswap32(881);
    char dName[30] = "Empty";
    root->name_len = strlen(dName);
    memcpy(&root->diskname[0], &dName[0], strlen(dName));
    root->extension = 0;
    root->sec_type = __bswap32(1);
    uint32_t newsum = 0;
    root->chksum = 0;
    for (int i = 0; i < 128; i++)
      newsum += rootTemp->data[i * 4] << 24
                | rootTemp->data[i * 4 + 1] << 16
                | rootTemp->data[i * 4 + 2] << 8
                | rootTemp->data[i * 4 + 3];
    newsum = -newsum;
    root->chksum = __bswap32(newsum);
    // construction of bitmap block
    struct Sector *bitmapTemp = (Sector *) &track[1].sector;
    struct bitmapBlock *bitmap = (bitmapBlock *) &bitmapTemp->data;
    if (densMode == HD) {
      for (int i = 0; i < 110; i++) {
        bitmap->map[i] = 0xffffffff;
      }
      bitmap->map[54] = __bswap32(0x3fffffff);
    } else {
      for (int i = 0; i < 55; i++) {
        bitmap->map[i] = 0xffffffff;
      }
      bitmap->map[27] = __bswap32(0xffff3fff);
    }
    newsum = 0;
    for (int i = 1; i < 128; i++)
      newsum -= bitmapTemp->data[i * 4] << 24
                | bitmapTemp->data[i * 4 + 1] << 16
                | bitmapTemp->data[i * 4 + 2] << 8
                | bitmapTemp->data[i * 4 + 3];
    bitmap->checksum = __bswap32(newsum);
  }
  Serial.printf("Track generated in %dms\n", millis() - zeit);
  int ret = 0;
  if (verify) ret = writeWithVerify(tTrack, wRetries);
  else ret = writeWithoutVerify(tTrack, wRetries);
  return ret;
}

/*
   formats a whole disk in OFS
*/
int formatDisk(boolean quick, boolean verify)
{
  int errors = 0;
  motorOn();
  if (!writeProtect())
  {
    if (quick) {
      formatTrack(0, 6, verify);
      formatTrack(80, 6, verify);
    } else
      for (int i = 0; i < 160; i++) {
        errors += formatTrack(i, 6, verify);
        if (errors != 0) return -1;
      }
  } else {
#ifdef debug
    Serial.println("Disk ist write protected.");
#endif
    return 1;
  }
  return errors;
}

/*
   write a track with verify
*/
int writeWithVerify(int tTrack, int wRetries)
{
  int error = 0;
  motorOn();
  floppyTrackMfmEncode(tTrack, (byte*)track, stream);
  gotoLogicTrack(tTrack);
  for (int i = 0; i < wRetries; i++) {
    writeTrack(preErase);
    error = readTrack(true, true);
#ifdef debug
    Serial.printf("%d ", error);
#endif
    if (error == 0) return 0;
    floppyTrackMfmEncode(tTrack, (byte*)track, stream);
  }
  if (error != 0) return 1;
  else return 0;
}

/*
   write a track without verify
*/
int writeWithoutVerify(int tTrack, int wRetries)
{
  int error = 0;
  uint32_t zeit = millis();
  motorOn();
  Serial.printf("Motor on: %d ", millis() - zeit);
  zeit = millis();
  floppyTrackMfmEncode(tTrack, (byte*)track, stream);
  Serial.printf(" Encode: %d ", millis() - zeit);
  zeit = millis();
  gotoLogicTrack(tTrack);
  Serial.printf(" goto Track: %d ", millis() - zeit);
  zeit = millis();
  writeTrack(preErase);
  Serial.printf(" Write: %d\n", millis() - zeit);
  if (error != 0) return 1;
  else return 0;
}

void printTimings(struct tSettings settings)
{
  Serial.print("Motor Spinup Delay: "); Serial.println(settings.motorSpinupDelay);
  Serial.print("Motor Spindown Delay: "); Serial.println(settings.motorSpindownDelay);
  Serial.print("Drive Select Delay: "); Serial.println(settings.driveSelectDelay);
  Serial.print("Drive Deselect Delay: "); Serial.println(settings.driveDeselectDelay);
  Serial.print("Set Direction Delay: "); Serial.println(settings.setDirDelay);
  Serial.print("Set Side Delay: "); Serial.println(settings.setSideDelay);
  Serial.print("Single Step Delay: "); Serial.println(settings.stepDelay);
  Serial.print("Step Settle Delay: "); Serial.println(settings.stepSettleDelay);
  Serial.print("gotoTrack Settle Delay: "); Serial.println(settings.gotoTrackSettle);
  Serial.print("sdRetries: "); Serial.println(settings.sdRetries);
  Serial.print("hdRetries: "); Serial.println(settings.hdRetries);
  Serial.print("config: 0b"); Serial.println(settings.config, BIN);
  Serial.print("reserved3: 0x"); Serial.println(settings.reserved3, HEX);
  Serial.print("reserved4: 0x"); Serial.println(settings.reserved4, HEX);
  Serial.print("reserved5: 0x"); Serial.println(settings.reserved5, HEX);
  Serial.print("reserved6: 0x"); Serial.println(settings.reserved6, HEX);
  Serial.print("reserved7: 0x"); Serial.println(settings.reserved7, HEX);
}

void setDefaultTimings()
{
  settings.motorSpinupDelay = 600; // ms
  settings.motorSpindownDelay = 50; // µs
  settings.driveSelectDelay = 100; // µs
  settings.driveDeselectDelay = 5; // µs
  settings.setDirDelay = 5; // µs
  settings.setSideDelay = 200; // µs
  settings.stepDelay = 2; // µs
  settings.stepSettleDelay = 3; // ms
  settings.gotoTrackSettle = 18; // ms
  settings.sdRetries = 10;
  settings.hdRetries = 30;
  settings.config = mtpModeOn;
  settings.reserved3 = 0xDEADDA7A;
  settings.reserved4 = 0xDEADDA7A;
  settings.reserved5 = 0xDEADDA7A;
  settings.reserved6 = 0xDEADDA7A;
  settings.reserved7 = 0xDEADDA7A;
}

void storeTimings()
{
  settings.versionNr = numVersion;
  settings.reserved3 = 0xDEADDA7A;
  settings.reserved4 = 0xDEADDA7A;
  settings.reserved5 = 0xDEADDA7A;
  settings.reserved6 = 0xDEADDA7A;
  settings.reserved7 = 0xDEADDA7A;
  settings.checksum = 0x00000000;
  uint32_t *settingsArray = &(settings.versionNr);
  uint32_t checksum = 0xC0FFEE;
  for (int i = 0; i < 18; i++) {
    checksum += settingsArray[i];
  }
#ifdef debug
  Serial.printf("Checksum: %lx\n", checksum);
#endif
  settings.checksum = checksum;
  EEPROM.put(0, settings);
}

void storeDefaultTimings()
{
  setDefaultTimings();
  storeTimings();
}

int restoreTimings()
{
  tSettings tempSettings;
  EEPROM.get(0, tempSettings);
  tempSettings.versionNr = numVersion;
  uint32_t *settingsArray = &(tempSettings.versionNr);
  uint32_t checksum = 0xC0FFEE;
  for (int i = 0; i < 18; i++) {
    checksum += settingsArray[i];
  }
#ifdef debug
  Serial.printf("Checksum calculated: %lx\n", checksum);
  Serial.printf("Checksum in EEPROM: %lx\n", tempSettings.checksum);
  printTimings(tempSettings);
#endif
  if (tempSettings.checksum == checksum) {
    settings = tempSettings;
    if ((settings.config & 0x01) == 1) mtpOn = true;
    else mtpOn = false;
    return 0;
  } else {
    return -1;
  }
}

/*
   the ISR for the blinking errorcodes
*/
#ifdef errLed
void errorLed()
{
  if (failureCode == 0) {
    digitalWriteFast(13, LOW);
    return;
  }
  blinkCode = ((blinkCode & 0x8000) ? 0x01 : 0x00) | (blinkCode << 1);
  digitalWriteFast(13, blinkCode & 0x0001);
}

void startErrorLed()
{
  pinMode(13, OUTPUT);
  digitalWrite(13, LOW);
  errorTimer.priority(255);
  errorTimer.begin(errorLed, 250000);
}

void stopErrorLed()
{
  errorTimer.end();
}
#endif
